// SPDX-License-Identifier: GPL-2.0
/*
 * uid_audit.c - Taobao security enhancement feature.
 *
 * The linux kernel may have many vulnerabilities that can
 * be exploited. The common exploit technology is set
 * the process's uid/euid/suid as 0.
 *
 * 1. We add config "CONFIG_DETECT_KERNEL_VUL" for
 * enabling the detection of kernel vulnerability attack.
 * When this feature is enabled, this patch will:
 * - set a uid_canary/euid_canary/suid_canary in the task_struct.
 * - check the [x]id and [x]id_canary when the process exited.
 * - print some messages or panic if the system is attacked.
 *
 * 2. Add /proc/sys/kernel/detect_kernel_vul which controls
 * both security check level and behavior if the system
 * is attacked.
 *
 * The detect_kernel_vul is 4-bit map in /proc/sys/kernel.
 * - Low 4-bit controls the behavior:
 *  0x01: only print warning messages
 *  0x02: kill the process
 *  0x04: cause kernel panic
 *
 *The default value is 0x1, implying all processes will
 *print warning messages if exploited.
 *
 *Usage examples:
 *- Apply to all processes and print warning messages
 *  echo 0x1 > /proc/sys/kernel/detect_kernel_vul
 *- Apply to all processes and cause kernel panic
 *  echo 0x4 > /proc/sys/kernel/detect_kernel_vul
 *- Disable this feature
 *  echo 0x00 > /proc/sys/kernel/detect_kernel_vul
 */

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/security.h>
#include <linux/mm.h>
#include <linux/threads.h>
#include <linux/nsproxy.h>
#include <linux/kref.h>
#include <linux/init_task.h>
#include <linux/pid_namespace.h>
#include <linux/capability.h>
#include <linux/uidgid.h>
#include <linux/uid_canary.h>

/* default to 0(KERNEL_VUL_DISABLED) */
int detect_kernel_vul;

static void uid_warning(const char *fmt, ...)
{
	char *buf;
	va_list args;

	buf = kmalloc(1024, GFP_KERNEL);
	if (buf == NULL)
		return;

	va_start(args, fmt);
	vsnprintf(buf, 1024, fmt, args);
	va_end(args);

	if (kernel_vul_detect(KERNEL_VUL_WARN)) {
		pr_warn("[Kernel Attack Warning] %s\n", buf);
	} else if (kernel_vul_detect(KERNEL_VUL_KILL)) {
		force_sig(SIGKILL);
		pr_warn("[Kernel Attack Warning] %s\n", buf);
	} else if (kernel_vul_detect(KERNEL_VUL_PANIC)) {
		panic("[Kernel Attack Warning] %s\n", buf);
	} else {
		pr_warn("[Kernel Attack Warning] bad parameters.\n");
	}

	kfree(buf);
}

void uid_canary_check(void)
{
	struct task_struct *tsk = current;
	const struct cred *cred = tsk->cred;

	if (kernel_vul_detect(KERNEL_VUL_DISABLED))
		return;

	/* uid canary has changed. */
	if (!uid_eq(cred->uid, tsk->uid_canary)) {
		uid_warning("task: %s pid: %d uid: %d euid %d suid: %d uid_canary: %d changed.",
			tsk->comm, tsk->pid, __kuid_val(cred->uid),
			__kuid_val(cred->euid), __kuid_val(cred->suid),
			__kuid_val(tsk->uid_canary));
		return;
	}

	/* euid canary has changed. */
	if (!uid_eq(cred->euid, tsk->euid_canary)) {
		uid_warning("task: %s pid: %d uid: %d euid %d suid: %d euid_canary: %d changed.",
			tsk->comm, tsk->pid, __kuid_val(cred->uid),
			__kuid_val(cred->euid), __kuid_val(cred->suid),
			 __kuid_val(tsk->euid_canary));
		return;
	}

	/* suid canary has changed. */
	if (!uid_eq(cred->suid, tsk->suid_canary)) {
		uid_warning("task: %s pid: %d uid: %d euid %d suid: %d suid_canary: %d changed.",
			tsk->comm, tsk->pid, __kuid_val(cred->uid),
			__kuid_val(cred->euid), __kuid_val(cred->suid),
			 __kuid_val(tsk->suid_canary));
		return;
	}
}

void uid_canary_set(const struct cred *cred, struct task_struct *p)
{
	p->uid_canary = cred->uid;
	p->euid_canary = cred->euid;
	p->suid_canary = cred->suid;
}
