// SPDX-License-Identifier: GPL-2.0
#ifndef __AEAD_H
#define __AEAD_H

#include <stdint.h>

#define CRYPTO_ASYNC			0x0
#define CRYPTO_SYNC			0x2

#define COMPLETION_TIMEOUT_SECS		5

struct aead_ctx {
	uint32_t flags; /* SYNC/ASYNC flags temporary */

	uint32_t authsize;
	uint32_t reqsize;

	struct aead_alg *alg;
	void *__ctx[];
};

struct aead_req;
typedef void (*aead_completion_t)(struct aead_req *req, int err);

struct aead_req {
	uint32_t cryptlen;
	uint32_t assoclen;
	unsigned char *iv;

	unsigned char *src;
	unsigned char *dst;

	aead_completion_t complete;
	void *data; /* May be used by completion */

	struct aead_ctx *ctx;
	void *__req[];
};

struct aead_alg {
	const char *name;
	int (*setkey)(struct aead_ctx *ctx, const char *key, uint32_t keylen);
	int (*setauthsize)(struct aead_ctx *ctx, unsigned int authsize);
	int (*encrypt)(struct aead_req *req);
	int (*decrypt)(struct aead_req *req);
	int (*init)(struct aead_ctx *ctx);
	void (*exit)(struct aead_ctx *ctx);
	uint32_t blocksize;
	uint32_t ivsize;
	uint32_t ctxsize;
	uint32_t maxauthsize;
};

struct aead_alg_entry {
	struct aead_alg *alg;
	struct aead_alg_entry *next;
};

struct aead_ctx *aead_alloc_ctx(const char *alg_name, uint32_t flag);
void aead_free_ctx(struct aead_ctx *ctx);
struct aead_req *aead_alloc_req(struct aead_ctx *ctx);
void aead_free_req(struct aead_req *req);
int aead_setkey(struct aead_ctx *ctx, const char *key, unsigned int keylen);
int aead_setauthsize(struct aead_ctx *ctx, unsigned int authsize);
void aead_req_set_callback(struct aead_req *req, aead_completion_t complete, void *data);
void aead_set_req(struct aead_req *req, unsigned char *src, unsigned char *dst,
		  unsigned int cryptlen, unsigned char *iv);
void aead_set_req_ad(struct aead_req *req, unsigned int assoclen);
int aead_encrypt(struct aead_req *req);
int aead_decrypt(struct aead_req *req);
void aead_register_algs(void);
void aead_unregister_algs(void);

#endif
