/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.bytes;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.Objects;
import java.util.stream.IntStream;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.BytesRefBuilder;
import org.apache.lucene.util.BytesRefIterator;
import org.apache.lucene.util.RamUsageEstimator;
import org.elasticsearch.common.bytes.AbstractBytesReference;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;

public final class CompositeBytesReference
extends AbstractBytesReference {
    private final BytesReference[] references;
    private final int[] offsets;
    private final int length;
    private final long ramBytesUsed;

    public static BytesReference of(BytesReference ... references) {
        if (references.length == 0) {
            return BytesArray.EMPTY;
        }
        if (references.length == 1) {
            return references[0];
        }
        return CompositeBytesReference.ofMultiple(references);
    }

    private static BytesReference ofMultiple(BytesReference[] references) {
        assert (references.length > 1) : "use #of() instead";
        int[] offsets = new int[references.length];
        long ramBytesUsed = 0L;
        int offset = 0;
        for (int i = 0; i < references.length; ++i) {
            BytesReference reference = references[i];
            if (reference == null) {
                throw new IllegalArgumentException("references must not be null");
            }
            if (reference.length() == 0) {
                return CompositeBytesReference.dropEmptyReferences(references);
            }
            offsets[i] = offset;
            if ((offset += reference.length()) <= 0) {
                throw new IllegalArgumentException("CompositeBytesReference cannot hold more than 2GB");
            }
            ramBytesUsed += reference.ramBytesUsed();
        }
        return new CompositeBytesReference(references, offsets, offset, ramBytesUsed + (long)(4 * offsets.length + RamUsageEstimator.NUM_BYTES_ARRAY_HEADER) + (long)(references.length * RamUsageEstimator.NUM_BYTES_OBJECT_REF + RamUsageEstimator.NUM_BYTES_ARRAY_HEADER) + 4L + 8L);
    }

    private static BytesReference dropEmptyReferences(BytesReference[] references) {
        BytesReference[] tempArray = new BytesReference[references.length];
        int targetIndex = 0;
        for (BytesReference reference : references) {
            if (reference.length() == 0) continue;
            tempArray[targetIndex++] = reference;
        }
        assert (targetIndex < references.length) : "no empty references found";
        BytesReference[] filteredReferences = new BytesReference[targetIndex];
        System.arraycopy(tempArray, 0, filteredReferences, 0, targetIndex);
        return CompositeBytesReference.of(filteredReferences);
    }

    private CompositeBytesReference(BytesReference[] references, int[] offsets, int length, long ramBytesUsed) {
        assert (references != null && offsets != null);
        assert (references.length > 1) : "Should not build composite reference from less than two references but received [" + references.length + "]";
        assert (Arrays.stream(references).allMatch(r -> r != null && r.length() > 0));
        assert (offsets[0] == 0);
        assert (IntStream.range(1, references.length).allMatch(i -> offsets[i] - offsets[i - 1] == references[i - 1].length()));
        assert ((long)length == Arrays.stream(references).mapToLong(BytesReference::length).sum());
        assert (ramBytesUsed > Arrays.stream(references).mapToLong(BytesReference::ramBytesUsed).sum());
        this.references = Objects.requireNonNull(references, "references must not be null");
        this.offsets = offsets;
        this.length = length;
        this.ramBytesUsed = ramBytesUsed;
    }

    @Override
    public byte get(int index) {
        int i = this.getOffsetIndex(index);
        return this.references[i].get(index - this.offsets[i]);
    }

    @Override
    public int indexOf(byte marker, int from) {
        int firstReferenceIndex;
        int remainingBytes = Math.max(this.length - from, 0);
        Objects.checkFromIndexSize(from, remainingBytes, this.length);
        int result = -1;
        if (this.length == 0) {
            return result;
        }
        for (int i = firstReferenceIndex = this.getOffsetIndex(from); i < this.references.length; ++i) {
            BytesReference reference = this.references[i];
            int internalFrom = i == firstReferenceIndex ? from - this.offsets[firstReferenceIndex] : 0;
            result = reference.indexOf(marker, internalFrom);
            if (result == -1) continue;
            result += this.offsets[i];
            break;
        }
        return result;
    }

    @Override
    public int length() {
        return this.length;
    }

    @Override
    public BytesReference slice(int from, int length) {
        if (from == 0 && this.length == length) {
            return this;
        }
        Objects.checkFromIndexSize(from, length, this.length);
        if (length == 0) {
            return BytesArray.EMPTY;
        }
        int to = from + length;
        int limit = this.getOffsetIndex(to - 1);
        int start = this.getOffsetIndex(from);
        BytesReference[] inSlice = new BytesReference[1 + (limit - start)];
        int j = start;
        for (int i = 0; i < inSlice.length; ++i) {
            inSlice[i] = this.references[j++];
        }
        int inSliceOffset = from - this.offsets[start];
        if (inSlice.length == 1) {
            return inSlice[0].slice(inSliceOffset, length);
        }
        inSlice[0] = inSlice[0].slice(inSliceOffset, inSlice[0].length() - inSliceOffset);
        inSlice[inSlice.length - 1] = inSlice[inSlice.length - 1].slice(0, to - this.offsets[limit]);
        return CompositeBytesReference.ofMultiple(inSlice);
    }

    private int getOffsetIndex(int offset) {
        int i = Arrays.binarySearch(this.offsets, offset);
        return i < 0 ? -(i + 1) - 1 : i;
    }

    @Override
    public BytesRef toBytesRef() {
        BytesRefBuilder builder = new BytesRefBuilder();
        builder.grow(this.length());
        BytesRefIterator iterator = this.iterator();
        try {
            BytesRef spare;
            while ((spare = iterator.next()) != null) {
                builder.append(spare);
            }
        }
        catch (IOException ex) {
            throw new AssertionError("won't happen", ex);
        }
        return builder.toBytesRef();
    }

    @Override
    public BytesRefIterator iterator() {
        return new BytesRefIterator(){
            int index = 0;
            private BytesRefIterator current;
            {
                this.current = CompositeBytesReference.this.references[this.index++].iterator();
            }

            public BytesRef next() throws IOException {
                BytesRef next = this.current.next();
                if (next == null) {
                    while (this.index < CompositeBytesReference.this.references.length) {
                        this.current = CompositeBytesReference.this.references[this.index++].iterator();
                        next = this.current.next();
                        if (next == null) continue;
                        break;
                    }
                }
                return next;
            }
        };
    }

    @Override
    public void writeTo(OutputStream os) throws IOException {
        for (BytesReference reference : this.references) {
            reference.writeTo(os);
        }
    }

    @Override
    public long ramBytesUsed() {
        return this.ramBytesUsed;
    }

    @Override
    public int getIntLE(int index) {
        BytesReference wholeIntLivesHere;
        int i = this.getOffsetIndex(index);
        int idx = index - this.offsets[i];
        int end = idx + 4;
        if (end <= (wholeIntLivesHere = this.references[i]).length()) {
            return wholeIntLivesHere.getIntLE(idx);
        }
        return super.getIntLE(index);
    }

    @Override
    public long getLongLE(int index) {
        BytesReference wholeLongsLivesHere;
        int i = this.getOffsetIndex(index);
        int idx = index - this.offsets[i];
        int end = idx + 8;
        if (end <= (wholeLongsLivesHere = this.references[i]).length()) {
            return wholeLongsLivesHere.getLongLE(idx);
        }
        return super.getLongLE(index);
    }

    @Override
    public double getDoubleLE(int index) {
        BytesReference wholeDoublesLivesHere;
        int i = this.getOffsetIndex(index);
        int idx = index - this.offsets[i];
        int end = idx + 8;
        if (end <= (wholeDoublesLivesHere = this.references[i]).length()) {
            return wholeDoublesLivesHere.getDoubleLE(idx);
        }
        return super.getDoubleLE(index);
    }
}

