/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.scanner.bootstrap;

import com.google.common.annotations.VisibleForTesting;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonar.api.config.Configuration;
import org.sonar.scanner.bootstrap.DefaultScannerWsClient;
import org.sonar.scanner.bootstrap.ScannerPluginInstaller;
import org.sonarqube.ws.client.GetRequest;
import org.sonarqube.ws.client.HttpException;
import org.sonarqube.ws.client.RequestWithoutPayload;
import org.sonarqube.ws.client.WsResponse;

public class PluginFiles {
    private static final Logger LOGGER = LoggerFactory.getLogger(PluginFiles.class);
    private static final String MD5_HEADER = "Sonar-MD5";
    @VisibleForTesting
    static final String PLUGINS_DOWNLOAD_TIMEOUT_PROPERTY = "sonar.plugins.download.timeout";
    private static final int PLUGINS_DOWNLOAD_TIMEOUT_DEFAULT = 300;
    private final DefaultScannerWsClient wsClient;
    private final Configuration configuration;
    private final File cacheDir;
    private final File tempDir;

    public PluginFiles(DefaultScannerWsClient wsClient, Configuration configuration) {
        this.wsClient = wsClient;
        this.configuration = configuration;
        File home = PluginFiles.locateHomeDir(configuration);
        this.cacheDir = PluginFiles.mkdir(new File(home, "cache"), "user cache");
        this.tempDir = PluginFiles.mkdir(new File(home, "_tmp"), "temp dir");
        LOGGER.info("User cache: {}", (Object)this.cacheDir.getAbsolutePath());
    }

    public File createTempDir() {
        try {
            return Files.createTempDirectory(this.tempDir.toPath(), "plugins", new FileAttribute[0]).toFile();
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to create temp directory in " + this.tempDir, e);
        }
    }

    public Optional<File> get(ScannerPluginInstaller.InstalledPlugin plugin) {
        File jarInCache = this.jarInCache(plugin.key, plugin.hash);
        if (jarInCache.exists() && jarInCache.isFile()) {
            return Optional.of(jarInCache);
        }
        return this.download(plugin);
    }

    private Optional<File> download(ScannerPluginInstaller.InstalledPlugin plugin) {
        GetRequest request = (GetRequest)((RequestWithoutPayload)new GetRequest("api/plugins/download").setParam("plugin", plugin.key)).setTimeOutInMs(this.configuration.getInt(PLUGINS_DOWNLOAD_TIMEOUT_PROPERTY).orElse(300) * 1000);
        File downloadedFile = this.newTempFile();
        LOGGER.debug("Download plugin '{}' to '{}'", (Object)plugin.key, (Object)downloadedFile);
        WsResponse response = this.wsClient.call(request);
        try {
            Optional<String> expectedMd5 = response.header(MD5_HEADER);
            if (!expectedMd5.isPresent()) {
                throw new IllegalStateException(String.format("Fail to download plugin [%s]. Request to %s did not return header %s", plugin.key, response.requestUrl(), MD5_HEADER));
            }
            PluginFiles.downloadBinaryTo(plugin, downloadedFile, response);
            String effectiveTempMd5 = PluginFiles.computeMd5(downloadedFile);
            if (!expectedMd5.get().equals(effectiveTempMd5)) {
                throw new IllegalStateException(String.format("Fail to download plugin [%s]. File %s was expected to have checksum %s but had %s", plugin.key, downloadedFile, expectedMd5.get(), effectiveTempMd5));
            }
            File tempJar = downloadedFile;
            String cacheMd5 = expectedMd5.get();
            File jarInCache = this.jarInCache(plugin.key, cacheMd5);
            PluginFiles.mkdir(jarInCache.getParentFile());
            PluginFiles.moveFile(tempJar, jarInCache);
            Optional<File> optional = Optional.of(jarInCache);
            if (response != null) {
                response.close();
            }
            return optional;
        }
        catch (Throwable throwable) {
            try {
                if (response != null) {
                    try {
                        response.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (HttpException e) {
                if (e.code() == 404) {
                    return Optional.empty();
                }
                throw new IllegalStateException(String.format("Fail to download plugin [%s]. Request to %s returned code %d.", plugin.key, e.url(), e.code()));
            }
        }
    }

    private static void downloadBinaryTo(ScannerPluginInstaller.InstalledPlugin plugin, File downloadedFile, WsResponse response) {
        try (InputStream stream = response.contentStream();){
            FileUtils.copyInputStreamToFile(stream, downloadedFile);
        }
        catch (IOException e) {
            throw new IllegalStateException(String.format("Fail to download plugin [%s] into %s", plugin.key, downloadedFile), e);
        }
    }

    private File jarInCache(String pluginKey, String hash) {
        File hashDir = new File(this.cacheDir, hash);
        File file = new File(hashDir, String.format("sonar-%s-plugin.jar", pluginKey));
        if (!file.getParentFile().toPath().equals(hashDir.toPath())) {
            throw new IllegalStateException(String.format("Fail to download plugin [%s]. Key is not valid.", pluginKey));
        }
        return file;
    }

    private File newTempFile() {
        try {
            return File.createTempFile("fileCache", null, this.tempDir);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to create temp file in " + this.tempDir, e);
        }
    }

    private static String computeMd5(File file) {
        String string;
        BufferedInputStream fis = new BufferedInputStream(FileUtils.openInputStream(file));
        try {
            string = DigestUtils.md5Hex(fis);
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((InputStream)fis).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new IllegalStateException("Fail to compute md5 of " + file, e);
            }
        }
        ((InputStream)fis).close();
        return string;
    }

    private static void moveFile(File sourceFile, File targetFile) {
        boolean rename = sourceFile.renameTo(targetFile);
        if (!rename && !targetFile.exists()) {
            LOGGER.warn("Unable to rename {} to {}", (Object)sourceFile.getAbsolutePath(), (Object)targetFile.getAbsolutePath());
            LOGGER.warn("A copy/delete will be tempted but with no guarantee of atomicity");
            try {
                Files.move(sourceFile.toPath(), targetFile.toPath(), new CopyOption[0]);
            }
            catch (IOException e) {
                throw new IllegalStateException("Fail to move " + sourceFile.getAbsolutePath() + " to " + targetFile, e);
            }
        }
    }

    private static void mkdir(File dir) {
        try {
            Files.createDirectories(dir.toPath(), new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to create cache directory: " + dir, e);
        }
    }

    private static File mkdir(File dir, String debugTitle) {
        if (!dir.isDirectory() || !dir.exists()) {
            LOGGER.debug("Create : {}", (Object)dir.getAbsolutePath());
            try {
                Files.createDirectories(dir.toPath(), new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new IllegalStateException("Unable to create " + debugTitle + dir.getAbsolutePath(), e);
            }
        }
        return dir;
    }

    private static File locateHomeDir(Configuration configuration) {
        return Stream.of(configuration.get("sonar.userHome").orElse(null), System.getenv("SONAR_USER_HOME"), System.getProperty("user.home") + File.separator + ".sonar").filter(Objects::nonNull).findFirst().map(File::new).get();
    }
}

