/* gssapi.c --- Test the GSSAPI mechanism.
 * Copyright (C) 2002-2024 Simon Josefsson
 *
 * This file is part of GNU SASL.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <config.h>

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#include "utils.h"

#define SERVICE "host"
#define HOST "latte.josefsson.org"
#define GSSAPI_USER "jas"

static const char *USERNAME[] = {
  "foo", "BABABA", "jas", "hepp", "@", NULL
};

size_t i;

static int
callback (Gsasl *ctx, Gsasl_session *sctx, Gsasl_property prop)
{
  int rc = GSASL_NO_CALLBACK;

  (void) ctx;

  switch (prop)
    {
    case GSASL_AUTHID:
      fail ("AUTHID should never be invoked by GSS-API mechanism!\n");
      break;

    case GSASL_AUTHZID:
      rc = gsasl_property_set (sctx, GSASL_AUTHZID, USERNAME[i]);
      break;

    case GSASL_SERVICE:
      rc = gsasl_property_set (sctx, prop, SERVICE);
      break;

    case GSASL_HOSTNAME:
      rc = gsasl_property_set (sctx, prop, HOST);
      break;

    case GSASL_VALIDATE_GSSAPI:
      {
	const char *client_name =
	  gsasl_property_fast (sctx, GSASL_GSSAPI_DISPLAY_NAME);
	const char *authzid = gsasl_property_fast (sctx, GSASL_AUTHZID);

	printf ("GSSAPI user: %s\n", client_name);
	printf ("Authorization ID: %s\n", authzid);

	if (strcmp (client_name, GSSAPI_USER) == 0 &&
	    ((USERNAME[i] == NULL && authzid == NULL)
	     || (USERNAME[i] != NULL && strcmp (authzid, USERNAME[i]) == 0)))
	  rc = GSASL_OK;
	else
	  rc = GSASL_AUTHENTICATION_ERROR;
      }
      break;

    default:
      fail ("Unknown callback property %u\n", prop);
      break;
    }

  return rc;
}

void
doit (void)
{
  Gsasl *ctx = NULL;
  Gsasl_session *server = NULL, *client = NULL;
  char *s1 = NULL, *s2 = NULL;
  int rc, res1, res2;

  if (getenv ("GNUGSS") == NULL || strcmp (getenv ("GNUGSS"), "yes") != 0)
    {
      fail ("Not using GNU GSS, skipping self-test.\n");
      exit (77);
    }

  rc = gsasl_init (&ctx);
  if (rc != GSASL_OK)
    {
      fail ("gsasl_init() failed (%d):\n%s\n", rc, gsasl_strerror (rc));
      return;
    }

  if (!gsasl_client_support_p (ctx, "GSSAPI")
      || !gsasl_server_support_p (ctx, "GSSAPI"))
    {
      gsasl_done (ctx);
      fail ("No support for GSSAPI.\n");
      exit (77);
    }

  gsasl_callback_set (ctx, callback);

  for (i = 0; i < 6; i++)
    {
      bool server_first = (i % 2) == 0;

      rc = gsasl_server_start (ctx, "GSSAPI", &server);
      if (rc != GSASL_OK)
	{
	  fail ("gsasl_server_start() failed (%d):\n%s\n",
		rc, gsasl_strerror (rc));
	  return;
	}
      rc = gsasl_client_start (ctx, "GSSAPI", &client);
      if (rc != GSASL_OK)
	{
	  fail ("gsasl_client_start() failed (%d):\n%s\n",
		rc, gsasl_strerror (rc));
	  return;
	}

      do
	{
	  res1 = gsasl_step64 (server_first ? server : client, s1, &s2);
	  if (s1)
	    {
	      gsasl_free (s1);
	      s1 = NULL;
	    }
	  if (res1 != GSASL_OK && res1 != GSASL_NEEDS_MORE)
	    {
	      fail ("gsasl_step64 (1) failed (%d):\n%s\n", res1,
		    gsasl_strerror (res1));
	      return;
	    }

	  if (debug)
	    printf ("%c: %s [%c]\n", server_first ? 'S' : 'C',
		    s2, res1 == GSASL_OK ? 'O' : 'N');

	  if (res1 == GSASL_OK && strcmp (s2, "") == 0)
	    {
	      gsasl_free (s2);
	      break;
	    }

	  res2 = gsasl_step64 (server_first ? client : server, s2, &s1);
	  gsasl_free (s2);
	  if (res2 != GSASL_OK && res2 != GSASL_NEEDS_MORE)
	    {
	      fail ("gsasl_step64 (2) failed (%d):\n%s\n", res2,
		    gsasl_strerror (res2));
	      /* likely gss without krb5 */
	      if (res2 == GSASL_GSSAPI_INIT_SEC_CONTEXT_ERROR)
		exit (77);
	      return;
	    }

	  if (debug)
	    printf ("%c: %s [%c]\n", server_first ? 'C' : 'S',
		    s1, res2 == GSASL_OK ? 'O' : 'N');
	}
      while (res1 != GSASL_OK || res2 != GSASL_OK);

      if (s1)
	{
	  gsasl_free (s1);
	  s1 = NULL;
	}

      if (debug)
	printf ("\n");

      gsasl_finish (client);
      gsasl_finish (server);
    }

  gsasl_done (ctx);
}
