// SPDX-License-Identifier: Apache-2.0

use super::{
    super::{
        nm_dbus::{NmConnection, NmIfaceType, NmSettingVeth},
        NmConnectionMatcher,
    },
    connection::gen_nm_conn_setting,
    ip::gen_nm_ip_setting,
};
use crate::{
    BaseInterface, EthernetInterface, Interface, InterfaceIpv4, InterfaceIpv6,
    InterfaceState, InterfaceType, NmstateError, VethConfig,
};

impl From<&VethConfig> for NmSettingVeth {
    fn from(config: &VethConfig) -> Self {
        let mut settings = NmSettingVeth::default();
        settings.peer = Some(config.peer.to_string());
        settings
    }
}

pub(crate) fn create_veth_peer_connection_if_not_found(
    peer_name: &str,
    end_name: &str,
    conn_matcher: &NmConnectionMatcher,
    stable_uuid: bool,
) -> Result<NmConnection, NmstateError> {
    if let Some(nm_conn) = conn_matcher
        .get_prefered_saved_by_name_type(peer_name, &NmIfaceType::Ethernet)
    {
        return Ok(nm_conn.clone());
    }
    // Create new connection
    let mut eth_iface = EthernetInterface::new();
    eth_iface.base = BaseInterface {
        name: peer_name.to_string(),
        iface_type: InterfaceType::Veth,
        state: InterfaceState::Up,
        ipv4: Some(InterfaceIpv4::new()),
        ipv6: Some(InterfaceIpv6::new()),
        ..Default::default()
    };
    let iface = Interface::Ethernet(Box::new(eth_iface));
    let mut nm_conn = NmConnection::default();
    gen_nm_conn_setting(&iface, &mut nm_conn, stable_uuid)?;
    gen_nm_ip_setting(&iface, None, &mut nm_conn)?;
    nm_conn.veth = Some(NmSettingVeth::from(&VethConfig {
        peer: end_name.to_string(),
    }));
    Ok(nm_conn)
}
