package manifest

import (
	"github.com/osbuild/osbuild-composer/internal/artifact"
	"github.com/osbuild/osbuild-composer/internal/osbuild"
)

// A VPC turns a raw image file into qemu-based image format, such as qcow2.
type VPC struct {
	Base
	Filename string

	ForceSize *bool

	imgPipeline *RawImage
}

// NewVPC createsa new Qemu pipeline. imgPipeline is the pipeline producing the
// raw image. The pipeline name is the name of the new pipeline. Filename is the name
// of the produced image.
func NewVPC(m *Manifest,
	buildPipeline *Build,
	imgPipeline *RawImage) *VPC {
	p := &VPC{
		Base:        NewBase(m, "vpc", buildPipeline),
		imgPipeline: imgPipeline,
		Filename:    "image.vhd",
	}
	if imgPipeline.Base.manifest != m {
		panic("live image pipeline from different manifest")
	}
	buildPipeline.addDependent(p)
	m.addPipeline(p)
	return p
}

func (p *VPC) serialize() osbuild.Pipeline {
	pipeline := p.Base.serialize()

	formatOptions := osbuild.VPCOptions{ForceSize: p.ForceSize}

	pipeline.AddStage(osbuild.NewQEMUStage(
		osbuild.NewQEMUStageOptions(p.Filename, osbuild.QEMUFormatVPC, formatOptions),
		osbuild.NewQemuStagePipelineFilesInputs(p.imgPipeline.Name(), p.imgPipeline.Filename),
	))

	return pipeline
}

func (p *VPC) getBuildPackages() []string {
	return []string{"qemu-img"}
}

func (p *VPC) Export() *artifact.Artifact {
	p.Base.export = true
	mimeType := "application/x-vhd"
	return artifact.New(p.Name(), p.Filename, &mimeType)
}
