// Copyright 2022 Contributors to the Veraison project.
// SPDX-License-Identifier: Apache-2.0

#[derive(thiserror::Error, PartialEq, Eq)]
pub enum Error {
    #[error("configuration error: {0}")]
    ConfigError(String),
    #[error("API error: {0}")]
    ApiError(String),
    #[error("callback error: {0}")]
    CallbackError(String),
    #[error("feature not implemented: {0}")]
    NotImplementedError(String),
    #[error("Data conversion error: {0}")]
    DataConversionError(String),
}

// While for other error sources the mapping may be more subtle, all reqwest
// errors are bottled as ApiErrors.
impl From<reqwest::Error> for Error {
    fn from(re: reqwest::Error) -> Self {
        Error::ApiError(re.to_string())
    }
}

impl From<jsonwebkey::ConversionError> for Error {
    fn from(e: jsonwebkey::ConversionError) -> Self {
        Error::DataConversionError(e.to_string())
    }
}

impl std::fmt::Debug for Error {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            Error::NotImplementedError(e)
            | Error::ConfigError(e)
            | Error::ApiError(e)
            | Error::CallbackError(e)
            | Error::DataConversionError(e) => {
                write!(f, "{}", e)
            }
        }
    }
}

/// EvidenceCreationCb is the function signature of the application callback.
/// The application is passed the session nonce and the list of supported
/// evidence media types and shall return the computed evidence together with
/// the selected media type.
type EvidenceCreationCb = fn(nonce: &[u8], accepted: &[String], token: Vec<u8>) -> Result<(Vec<u8>, String), Error>;

/// A builder for ChallengeResponse objects
pub struct ChallengeResponseBuilder {
    new_session_url: Option<String>,
    // TODO(tho) add TLS config / authn tokens etc.
}

impl ChallengeResponseBuilder {
    /// default constructor
    pub fn new() -> Self {
        Self {
            new_session_url: None,
        }
    }

    /// Use this method to supply the URL of the verification endpoint that will create
    /// new challenge-response sessions, e.g.
    /// "https://veraison.example/challenge-response/v1/newSession".
    pub fn with_new_session_url(mut self, v: String) -> ChallengeResponseBuilder {
        self.new_session_url = Some(v);
        self
    }

    /// Instantiate a valid ChallengeResponse object, or fail with an error.
    pub fn build(self) -> Result<ChallengeResponse, Error> {
        let new_session_url_str = self
            .new_session_url
            .ok_or_else(|| Error::ConfigError("missing API endpoint".to_string()))?;

        Ok(ChallengeResponse {
            new_session_url: url::Url::parse(&new_session_url_str)
                .map_err(|e| Error::ConfigError(e.to_string()))?,
            http_client: reqwest::Client::new(),
        })
    }
}

impl Default for ChallengeResponseBuilder {
    fn default() -> Self {
        Self::new()
    }
}

/// The object on which one or more challenge-response verification sessions can
/// be run.  Always use the [ChallengeResponseBuilder] to instantiate it.
pub struct ChallengeResponse {
    new_session_url: url::Url,
    http_client: reqwest::Client,
}

/// Nonce configuration: either the size (Size) of the nonce generated by the
/// server (use 0 to let the server also pick the size of the challenge), or an
/// explicit nonce (Value) supplied as a byte array.
pub enum Nonce {
    Size(usize),
    Value(Vec<u8>),
}

impl ChallengeResponse {
    /// Run a challenge-response verification session using the supplied nonce
    /// configuration and evidence creation callback. Returns the raw attestation results, or an
    /// error on failure.
    pub async fn run(
        &self,
        nonce: Nonce,
        evidence_creation_cb: EvidenceCreationCb,
        token: Vec<u8>,
    ) -> Result<String, Error> {
        // create new c/r verification session on the veraison side
        let (session_url, session) = self.new_session(&nonce).await?;

        // invoke the user-provided evidence builder callback with per-session parameters
        let (evidence, media_type) = (evidence_creation_cb)(session.nonce(), session.accept(), token)?;

        // send evidence for verification to the session endpoint
        let attestation_result = self.challenge_response(&evidence, &media_type, &session_url).await?;

        // return veraison's attestation results
        Ok(attestation_result)
    }

    /// Ask Veraison to create a new challenge/response session using the supplied nonce
    /// configuration. On success, the return value is a tuple of the session URL for subsequent
    /// operations, plus the session data including the nonce and the list of accept types.
    pub async fn new_session(&self, nonce: &Nonce) -> Result<(String, ChallengeResponseSession), Error> {
        // ask veraison for a new session object
        let resp = self.new_session_request(nonce).await.unwrap();

        // expect 201 and a Location header containing the URI of the newly
        // allocated session
        match resp.status() {
            reqwest::StatusCode::CREATED => (),
            status => {
                // on error the body is a RFC7807 problem detail
                //
                // NOTE(tho) -- this assumption does not hold in general because
                // the request may be intercepted (and dealt with) by HTTP
                // middleware that is unaware of the API.  We need something
                // more robust here that dispatches based on the Content-Type
                // header.
                let pd: ProblemDetails = resp.json().await?;

                return Err(Error::ApiError(format!(
                    "newSession response has unexpected status: {}.  Details: {}",
                    status, pd.detail
                )));
            }
        };

        // extract location header
        let loc = resp
            .headers()
            .get("location")
            .ok_or_else(|| {
                Error::ApiError("cannot determine URI of the session resource".to_string())
            })?
            .to_str()
            .map_err(|e| Error::ApiError(e.to_string()))?;

        // join relative location with base URI
        let session_url = resp
            .url()
            .join(loc)
            .map_err(|e| Error::ApiError(e.to_string()))?;

        // decode returned session object
        let crs: ChallengeResponseSession = resp.json().await?;

        Ok((session_url.to_string(), crs))
    }

    /// Execute a challenge/response operation with the given evidence.
    pub async fn challenge_response(
        &self,
        evidence: &[u8],
        media_type: &str,
        session_url: &str,
    ) -> Result<String, Error> {
        let c = &self.http_client;

        let resp = c
            .post(session_url)
            .header(reqwest::header::ACCEPT, CRS_MEDIA_TYPE)
            .header(reqwest::header::CONTENT_TYPE, media_type)
            .body(evidence.to_owned())
            .send()
            .await
            .unwrap();

        let status = resp.status();

        if status.is_success() {
            match status {
                reqwest::StatusCode::OK => {
                    let crs: ChallengeResponseSession = resp.json().await?;

                    if crs.status != "complete" {
                        return Err(Error::ApiError(format!(
                            "unexpected session state: {}",
                            crs.status
                        )));
                    }

                    let result = crs.result.ok_or_else(|| {
                        Error::ApiError(
                            "no attestation results found in completed session".to_string(),
                        )
                    })?;

                    Ok(result)
                }
                reqwest::StatusCode::ACCEPTED => {
                    // TODO(tho)
                    Err(Error::NotImplementedError("asynchronous model".to_string()))
                }
                status => Err(Error::ApiError(format!(
                    "session response has unexpected success status: {}",
                    status,
                ))),
            }
        } else {
            let pd: ProblemDetails = resp.json().await?;

            Err(Error::ApiError(format!(
                "session response has error status: {}.  Details: {}",
                status, pd.detail,
            )))
        }
    }

    async fn new_session_request(&self, nonce: &Nonce) -> Result<reqwest::Response, Error> {
        let u = self.new_session_request_url(nonce)?;

        let r = self
            .http_client
            .post(u.as_str())
            .header(reqwest::header::ACCEPT, CRS_MEDIA_TYPE)
            .send()
            .await
            .unwrap();

        Ok(r)
    }

    fn new_session_request_url(&self, nonce: &Nonce) -> Result<url::Url, Error> {
        let mut new_session_url = self.new_session_url.clone();

        let mut q_params = String::new();

        match nonce {
            Nonce::Value(val) if !val.is_empty() => {
                q_params.push_str("nonce=");
                q_params.push_str(&base64::encode_config(val, base64::URL_SAFE));
            }
            Nonce::Size(val) if *val > 0 => {
                q_params.push_str("nonceSize=");
                q_params.push_str(&val.to_string());
            }
            _ => {}
        }

        new_session_url.set_query(Some(&q_params));

        Ok(new_session_url)
    }
}

const CRS_MEDIA_TYPE: &str = "application/vnd.veraison.challenge-response-session+json";
const DISCOVERY_MEDIA_TYPE: &str = "application/vnd.veraison.discovery+json";

#[serde_with::serde_as]
#[serde_with::skip_serializing_none]
#[derive(serde::Deserialize, serde::Serialize)]
pub struct ChallengeResponseSession {
    #[serde_as(as = "serde_with::base64::Base64")]
    nonce: Vec<u8>,
    #[serde_as(as = "chrono::DateTime<chrono::Utc>")]
    expiry: chrono::NaiveDateTime,
    accept: Vec<String>,
    status: String,
    evidence: Option<EvidenceBlob>,
    result: Option<String>,
}

impl ChallengeResponseSession {
    pub fn nonce(&self) -> &[u8] {
        &self.nonce
    }

    pub fn accept(&self) -> &[String] {
        &self.accept
    }
}

#[serde_with::serde_as]
#[derive(serde::Deserialize, serde::Serialize)]
pub struct EvidenceBlob {
    r#type: String,
    #[serde_as(as = "serde_with::base64::Base64")]
    value: Vec<u8>,
}

/// Enumerates the four possible states that the service can be in.
#[derive(Debug, PartialEq, serde::Deserialize)]
#[serde(rename_all = "UPPERCASE")]
pub enum ServiceState {
    Down,
    Initializing,
    Ready,
    Terminating,
}

/// This object models the state and capabilities of the verification API in the Veraison service.
///
/// An instance of this struct is returned from [`Discovery::get_verification_api()`].
#[derive(serde::Deserialize)]
#[serde(rename_all = "kebab-case")]
pub struct VerificationApi {
    ear_verification_key: jsonwebkey::JsonWebKey,
    media_types: Vec<String>,
    version: String,
    //service_state: ServiceState,
    api_endpoints: std::collections::HashMap<String, String>,
}

impl VerificationApi {
    /// Obtains the EAR verification public key encoded in ASN.1 DER format.
    pub fn ear_verification_key_as_der(&self) -> Result<Vec<u8>, Error> {
        let key = &self.ear_verification_key.key;
        (*key)
            .try_to_der()
            .map_err(|e| Error::DataConversionError(e.to_string()))
    }

    /// Obtains the EAR verification public key encoded in PEM format.
    pub fn ear_verification_key_as_pem(&self) -> Result<String, Error> {
        let key = &self.ear_verification_key.key;
        (*key)
            .try_to_pem()
            .map_err(|e| Error::DataConversionError(e.to_string()))
    }

    /// Obtains the signature algorithm scheme used with the EAR.
    pub fn ear_verification_algorithm(&self) -> String {
        match &self.ear_verification_key.algorithm {
            Some(alg) => match alg {
                jsonwebkey::Algorithm::ES256 => String::from("ES256"),
                jsonwebkey::Algorithm::HS256 => String::from("HS256"),
                jsonwebkey::Algorithm::RS256 => String::from("RS256"),
            },
            None => String::from(""),
        }
    }

    /// Obtains the strings for the set of media types that are supported for evidence
    /// verification. Each member of the array will be a media type string such as
    /// `"application/eat-cwt; profile=http://arm.com/psa/2.0.0"`.
    pub fn media_types(&self) -> &[String] {
        self.media_types.as_ref()
    }

    /// Obtains the version of the service.
    pub fn version(&self) -> &str {
        self.version.as_ref()
    }

    /// Indicates whether the service is starting, ready, terminating or down.
    pub fn service_state(&self) -> &ServiceState {
        &ServiceState::Ready
    }

    /// Gets the API endpoint associated with a specific endpoint name.
    ///
    /// Returns `None` if there is no API endpoint with the given name, otherwise returns
    /// a relative URL such as `"/challenge-response/v1/newSession"`.
    pub fn get_api_endpoint(&self, endpoint_name: &str) -> Option<String> {
        self.api_endpoints.get(endpoint_name).cloned()
    }

    /// Gets all of the API endpoints published by this verification service as a vector of
    /// string pairs.
    ///
    /// For each endpoint entry, the first member of the pair is the endpoint name, such
    /// as `"newChallengeResponseSession"`, and the second member is the corresponding
    /// relative URL, such as `"/challenge-response/v1/newSession"`.
    pub fn get_all_api_endpoints(&self) -> Vec<(String, String)> {
        self.api_endpoints
            .iter()
            .map(|(k, v)| (k.clone(), v.clone()))
            .collect()
    }
}

/// This structure allows Veraison endpoints and service capabilities to be discovered
/// dynamically.
///
/// Use [`Discovery::from_base_url()`] to create an instance of this structure for the
/// Veraison service instance that you are communicating with.
pub struct Discovery {
    provisioning_url: url::Url, //TODO: The provisioning URL discovery is not implemented yet.
    verification_url: url::Url,
    http_client: reqwest::Client,
}

impl Discovery {
    /// Establishes client API discovery for the Veraison service instance running at the
    /// given base URL.
    pub fn from_base_url(base_url_str: String) -> Result<Discovery, Error> {
        let base_url =
            url::Url::parse(&base_url_str).map_err(|e| Error::ConfigError(e.to_string()))?;

        let mut provisioning_url = base_url.clone();
        provisioning_url.set_path(".well-known/veraison/provisioning");

        let mut verification_url = base_url;
        verification_url.set_path(".well-known/veraison/verification");

        Ok(Discovery {
            provisioning_url,
            verification_url,
            http_client: reqwest::Client::new(),
        })
    }

    /// Obtains the capabilities and endpoints of the Veraison verification service.
    pub async fn get_verification_api(&self) -> Result<VerificationApi, Error> {
        let response = self
            .http_client
            .get(self.verification_url.as_str())
            .header(reqwest::header::ACCEPT, DISCOVERY_MEDIA_TYPE)
            .send()
            .await
            .unwrap();

        match response.status() {
            reqwest::StatusCode::OK => Ok(response.json::<VerificationApi>().await?),
            _ => Err(Error::ApiError(String::from(
                "Failed to discover verification endpoint information.",
            ))),
        }
    }
}

#[derive(serde::Deserialize)]
struct ProblemDetails {
    r#type: String,
    title: String,
    status: u16,
    detail: String,
}

#[cfg(test)]
mod tests {
    use super::*;
    use wiremock::matchers::{method, path};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    const TEST_NEW_SESSION_URL_OK: &str =
        "https://veraison.example/challenge-response/v1/newSession";
    const TEST_NEW_SESSION_URL_NOT_ABSOLUTE: &str = "/challenge-response/v1/newSession";

    #[test]
    fn default_constructor() {
        let b: ChallengeResponseBuilder = Default::default();

        // expected initial state
        assert!(b.new_session_url.is_none());
    }

    #[test]
    fn build_ok() {
        let b = ChallengeResponseBuilder::new()
            .with_new_session_url(TEST_NEW_SESSION_URL_OK.to_string());

        assert!(b.build().is_ok());
    }

    #[test]
    fn build_fail_base_url_not_absolute() {
        let b = ChallengeResponseBuilder::new()
            .with_new_session_url(TEST_NEW_SESSION_URL_NOT_ABSOLUTE.to_string());

        assert!(b.build().is_err());
    }

    #[test]
    fn build_fail_missing_base_url() {
        let b = ChallengeResponseBuilder::new();

        assert!(b.build().is_err());
    }

    #[test]
    fn build_fail_missing_evidence_creation_cb() {
        let b = ChallengeResponseBuilder::new()
            .with_new_session_url(TEST_NEW_SESSION_URL_NOT_ABSOLUTE.to_string());

        assert!(b.build().is_err());
    }

    #[async_std::test]
    async fn new_session_request_ok() {
        let mock_server = MockServer::start().await;
        let nonce_value = vec![0xde, 0xad, 0xbe, 0xef, 0xde, 0xad, 0xbe, 0xef];
        let nonce = Nonce::Value(nonce_value.clone());

        let response = ResponseTemplate::new(201)
            .insert_header("location", "1234")
            .set_body_json(ChallengeResponseSession {
                nonce: nonce_value,
                status: "waiting".to_string(),
                accept: vec!["application/vnd.1".to_string()],
                evidence: None,
                result: None,
                expiry: chrono::Utc::now().naive_utc(),
            });

        Mock::given(method("POST"))
            .and(path("/newSession"))
            .respond_with(response)
            .mount(&mock_server)
            .await;

        let cr = ChallengeResponseBuilder::new()
            .with_new_session_url(mock_server.uri() + "/newSession")
            .build()
            .unwrap();

        let rv = cr.new_session(&nonce).await.expect("unexpected failure");

        // Expect we are given the expected location URL
        assert_eq!(rv.0, format!("{}/1234", mock_server.uri()));
    }

    #[async_std::test]
    async fn challenge_response_ok() {
        let mock_server = MockServer::start().await;
        let nonce_value = vec![0xbe, 0xef];
        let evidence_value: Vec<u8> = vec![0, 1];
        let evidence = EvidenceBlob {
            r#type: "application/vnd.1".to_string(),
            value: evidence_value.clone(),
        };
        let attestation_result = "a.b.c".to_string();

        let response = ResponseTemplate::new(200).set_body_json(ChallengeResponseSession {
            nonce: nonce_value,
            status: "complete".to_string(),
            accept: vec!["application/vnd.1".to_string()],
            evidence: Some(evidence),
            result: Some(attestation_result.clone()),
            expiry: chrono::Utc::now().naive_utc(),
        });

        Mock::given(method("POST"))
            .and(path("/session/5678"))
            .respond_with(response)
            .mount(&mock_server)
            .await;

        let cr = ChallengeResponseBuilder::new()
            .with_new_session_url(mock_server.uri() + "/newSession")
            .build()
            .unwrap();

        let session_url = mock_server.uri() + "/session/5678";
        let media_type = "application/vnd.1";

        let rv = cr
            .challenge_response(&evidence_value, media_type, &session_url)
            .await
            .expect("unexpected failure");

        // Expect we are given the expected attestation result
        assert_eq!(rv, attestation_result)
    }

    #[async_std::test]
    async fn discover_verification_ok() {
        let mock_server = MockServer::start().await;

        // Sample response crafted from Veraison docs.
        let raw_response = r#"
        {
            "ear-verification-key": {
                "crv": "P-256",
                "kty": "EC",
                "x": "usWxHK2PmfnHKwXPS54m0kTcGJ90UiglWiGahtagnv8",
                "y": "IBOL-C3BttVivg-lSreASjpkttcsz-1rb7btKLv8EX4",
                "alg": "ES256"
            },
            "media-types": [
                "application/eat-cwt; profile=http://arm.com/psa/2.0.0",
                "application/pem-certificate-chain",
                "application/vnd.enacttrust.tpm-evidence",
                "application/eat-collection; profile=http://arm.com/CCA-SSD/1.0.0",
                "application/psa-attestation-token"
            ],
            "version": "commit-cb11fa0",
            "service-state": "READY",
            "api-endpoints": {
                "newChallengeResponseSession": "/challenge-response/v1/newSession"
            }
        }"#;

        let response = ResponseTemplate::new(200)
            .set_body_raw(raw_response, "application/vnd.veraison.discovery+json");

        Mock::given(method("GET"))
            .and(path("/.well-known/veraison/verification"))
            .respond_with(response)
            .mount(&mock_server)
            .await;

        let discovery = Discovery::from_base_url(mock_server.uri())
            .expect("Failed to create Discovery client.");

        let verification_api = discovery
            .get_verification_api()
            .await
            .expect("Failed to get verification endpoint details.");

        // Check that we've pulled and deserialized everything that we expect
        //assert_eq!(verification_api.service_state, ServiceState::Ready);
        assert_eq!(verification_api.version, String::from("commit-cb11fa0"));
        assert_eq!(verification_api.media_types.len(), 5);
        assert_eq!(
            verification_api.media_types[0],
            String::from("application/eat-cwt; profile=http://arm.com/psa/2.0.0")
        );
        assert_eq!(
            verification_api.media_types[1],
            String::from("application/pem-certificate-chain")
        );
        assert_eq!(
            verification_api.media_types[2],
            String::from("application/vnd.enacttrust.tpm-evidence")
        );
        assert_eq!(
            verification_api.media_types[3],
            String::from("application/eat-collection; profile=http://arm.com/CCA-SSD/1.0.0")
        );
        assert_eq!(
            verification_api.media_types[4],
            String::from("application/psa-attestation-token")
        );
        assert_eq!(verification_api.api_endpoints.len(), 1);
        assert_eq!(
            verification_api
                .api_endpoints
                .get("newChallengeResponseSession"),
            Some(&String::from("/challenge-response/v1/newSession"))
        );
    }
}
