# Copyright (C) 2021 OpenAnolis Community
#
#
import re
from pyanaconda.flags import flags
from pyanaconda.ui.categories.software import SoftwareCategory
from pyanaconda.ui.tui.spokes import NormalTUISpoke
from pyanaconda.threading import threadMgr, AnacondaThread
from pyanaconda.payload import DependencyError, PackagePayload, payloadMgr, NoSuchGroup
from pyanaconda.core.i18n import N_, _, C_

from pyanaconda.core.constants import THREAD_PAYLOAD, THREAD_CHECK_SOFTWARE, \
                                      THREAD_SOFTWARE_WATCHER

from simpleline.render.containers import ListColumnContainer
from simpleline.render.screen import InputState
from simpleline.render.screen_handler import ScreenHandler
from simpleline.render.widgets import TextWidget, CheckboxWidget

from pyanaconda.anaconda_loggers import get_module_logger
log = get_module_logger(__name__)

__all__ = ["KernelSpoke"]


class KernelSpoke(NormalTUISpoke):
    """ Spoke used to read new value of text to represent source repo.

       .. inheritance-diagram:: SoftwareSpoke
          :parts: 3
    """
    help_id = "KernelSelectionSpoke"
    category = SoftwareCategory

    def __init__(self, data, storage, payload, instclass):
        super().__init__(data, storage, payload, instclass)
        self.title = N_("Kernel selection")
        self._container = None
        self.errors = []

        self._orig_kernel = None
        self._kickstarted = flags.automatedInstall and self.data.packages.seen

    def initialize(self):
        self.initialize_start()
        super().initialize()
        threadMgr.add(AnacondaThread(name="AnaKernelWatcher", target=self._initialize))

    def _initialize(self):
        threadMgr.wait(THREAD_PAYLOAD)

        self.payload.detectMultiKernel()

        # find kernel in package list
        kernel_pattern=re.compile(r"kernel-[4,5].\d+.\d+")
        if self._kickstarted:
            for package in self.payload.data.packages.packageList:
                if kernel_pattern.search(package):
                    self.current_kernel = package.split('-')[1]

        # use first kernel as default, normally it's rhck
        if not self.current_kernel:
            self.current_kernel = list(self.available_kernels.keys())[0]
 
        self._apply()
        threadMgr.wait("AnaCheckKernel")
        self.initialize_done()

    @property
    def changed(self):
        if self.current_kernel == self._orig_kernel:
            return False
        else:
            return True

    @property
    def showable(self):
        return isinstance(self.payload, PackagePayload)

    @property
    def status(self):
        if self.errors:
            return _("Error checking kernel selection")
        return self.current_kernel

    @property
    def completed(self):
        processing_done = self.ready and not self.errors
        return processing_done and self.payload.baseRepo

    def refresh(self, args=None):
        """ Refresh screen. """
        NormalTUISpoke.refresh(self, args)

        threadMgr.wait(THREAD_PAYLOAD)
        self._container = None

        threadMgr.wait("AnaCheckKernel")
        self._container = ListColumnContainer(2, columns_width=38, spacing=2)

        msg = self._refresh_kernels()

        self.window.add_with_separator(TextWidget(msg))
        self.window.add_with_separator(self._container)

    def _refresh_kernels(self):
        kernel_keys = self.available_kernels.keys()
        for kernel in kernel_keys:
            selected = (self.current_kernel == kernel)
            if kernel == '4.18.0':
                kernel_desc = _("RHCK")
                kernel_info = _("Compatible with RHEL")
            elif kernel == '4.19.91':
                kernel_desc = _("ANCK")
                kernel_info = _("Support Anolis OS verified platform")
            else:
                kernel_desc = ''
                kernel_info = ''
            widget = CheckboxWidget(title="%s (%s)" % (kernel, kernel_desc), completed=selected)
            self._container.add(widget, callback=self._set_kernel_callback, data=kernel)

        return _("Kernel List")

    def _set_kernel_callback(self, data):
        self.current_kernel = data

    def input(self, args, key):
        if self._container is not None and self._container.process_user_input(key):
            self.redraw()
        else:
            # TRANSLATORS: 'c' to continue
            if key.lower() == C_('TUI|Spoke Navigation', 'c'):

                if self.current_kernel is None:
                    self.close()
                else:
                    self.apply()
                    self.close()
            else:
                return super().input(args, key)

        return InputState.PROCESSED

    @property
    def ready(self):
        """ If we're ready to move on. """
        return (not threadMgr.get(THREAD_PAYLOAD) and
                not threadMgr.get("AnaCheckKernel") and
                not threadMgr.get("AnaKernelWatcher"))

    @property
    def available_kernels(self):
        return self.payload.available_kernels

    @property
    def current_kernel(self):
        return self.payload.current_kernel

    @current_kernel.setter
    def current_kernel(self, value):
        self.payload.current_kernel = value

    def apply(self):
        self._apply()

    def _apply(self):
        threadMgr.add(AnacondaThread(name="AnaCheckKernel", target=self.checkKernelSelection))

    def checkKernelSelection(self):
        if self.changed:
            kernel_keys = self.available_kernels.keys()
            for kernel in kernel_keys:
                if kernel == self.current_kernel:
                    self.payload.data.packages.packageList.extend(self.available_kernels[kernel])
                    for package in self.available_kernels[kernel]:
                        if package in self.payload.data.packages.excludedList:
                            self.payload.data.packages.excludedList.remove(package)
                else:
                    self.payload.data.packages.excludedList.extend(self.available_kernels[kernel])
                    for package in self.available_kernels[kernel]:
                        if package in self.payload.data.packages.packageList:
                            self.payload.data.packages.packageList.remove(package)
            # store kernel selection
            self._orig_kernel = self.current_kernel
