/* pbm2ppa.c
 * program to print a 600 dpi PBM file on the HP DJ820Cse or the HP720 series.
 * Copyright (c) 1998 Tim Norman.  See LICENSE for details
 * 2-24-98
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "pbm.h"
#include "ppa.h"

#include "defaults.h"

/* Paper sizes in 600ths of an inch. */

/* US Letter is 8.5 in by 11 in */

#define LETTERWIDTH  (5100)
#define LETTERHEIGHT (6600)

/* US Legal is 8.5 in by 14 in */

#define LEGALWIDTH  (5100)
#define LEGALHEIGHT (8400)

/* A4 is 210 mm by 297 mm == 8.27 in by 11.69 in */

#define A4WIDTH  (4960)
#define A4HEIGHT (7016)

int Width;      /* width and height in 600ths of an inch */
int Height;
int Pwidth;     /* width in bytes */

#define MAX_LINES 300

int cut_pbm_swath(pbm_stat* pbm,ppa_stat* prn,int maxlines,ppa_sweep_data* sweep_data);

ppa_stat printer;

int print_pbm (FILE *in)
{
  char line[1024];
  pbm_stat pbm;
  int done_page, numpages = 0;
  ppa_sweep_data sweeps[2];
  int current_sweep, previous_sweep;

  ppa_init_job(&printer);

  while(make_pbm_stat(&pbm,in))
  {
    ppa_init_page(&printer);
    ppa_load_page(&printer);

    sweeps[0].direction = right_to_left;
    sweeps[0].next=&sweeps[1];
    sweeps[1].direction = left_to_right;
    sweeps[1].next=&sweeps[0];

    current_sweep=0;
    previous_sweep=-1;

    done_page=0;
    while(!done_page)
      switch(cut_pbm_swath(&pbm,&printer,MAX_LINES,&sweeps[current_sweep]))
      {
      case 0:
	fprintf (stderr, "print_pbm(): error calling cut_pbm_swath()\n");
	return 1;
      case 1:
	done_page=1;
	break;
      case 2:
	if(previous_sweep>=0)
	{
	  ppa_print_sweep(&printer,&sweeps[previous_sweep]);
	  free(sweeps[previous_sweep].image_data);
	  free(sweeps[previous_sweep].nozzle_data);
	}
	previous_sweep=current_sweep;
	current_sweep= current_sweep==0 ? 1 : 0;
	break;
      default:
	fprintf(stderr,"print_pbm(): unknown return code from cut_pbm_swath()\n");
	return 1;
      }

    if(previous_sweep>=0)
    {
      sweeps[previous_sweep].next=NULL;
      ppa_print_sweep(&printer,&sweeps[previous_sweep]);
    }

    free(sweeps[0].image_data);
    free(sweeps[0].nozzle_data);
    free(sweeps[1].image_data);
    free(sweeps[1].nozzle_data);

    ppa_eject_page(&printer);

    /* eat any remaining whitespace */
    if(pbm.version==P1)
      fgets (line, 1024, in);

    numpages++;
  }

  if (numpages == 0)
  {
    fprintf (stderr, "No pages printed!\n");
    return 1;
  }

  ppa_end_print(&printer);

  fclose (pbm.fptr);
  fclose (printer.fptr);

  return 0;
}

void set_printer_specific_defaults()
{
  switch(printer.version)
  {
  case HP720:
    printer.x_offset=HP720_X_OFFSET;
    printer.y_offset=HP720_Y_OFFSET;
    printer.top_margin=HP720_TOP_MARGIN;
    printer.left_margin=HP720_LEFT_MARGIN;
    printer.right_margin=HP720_RIGHT_MARGIN;
    printer.bottom_margin=HP720_BOTTOM_MARGIN;
    printer.marg_diff=2;
    printer.bufsize = 200*1024;
    break;
  case HP820:
    printer.x_offset=HP820_X_OFFSET;
    printer.y_offset=HP820_Y_OFFSET;
    printer.top_margin=HP820_TOP_MARGIN;
    printer.left_margin=HP820_LEFT_MARGIN;
    printer.right_margin=HP820_RIGHT_MARGIN;
    printer.bottom_margin=HP820_BOTTOM_MARGIN;
    printer.marg_diff=0x62;
    printer.bufsize = 100*1024;
    break;
  case HP1000:
    printer.x_offset=HP1000_X_OFFSET;
    printer.y_offset=HP1000_Y_OFFSET;
    printer.top_margin=HP1000_TOP_MARGIN;
    printer.left_margin=HP1000_LEFT_MARGIN;
    printer.right_margin=HP1000_RIGHT_MARGIN;
    printer.bottom_margin=HP1000_BOTTOM_MARGIN;
    printer.marg_diff=0x62;
    printer.bufsize = 100*1024;
    break;
  default:
    fprintf(stderr,"set_printer_defaults(): unknown printer version\n");
    exit(1);
  }
}

void show_usage(char* prog)
{
  printf("usage: %s [ options ] [ <infile> [ <outfile> ] ]\n\n",prog);
  printf("  Prints a pbm- or pbmraw-format <infile> to HP720/820/1000-format <outfile>.\n\n");
  printf("    -b <margin>    bottom margin in 1\"/600 (default: 150 = 0.25\")\n");
  printf("    -d              dumps the configuration to stdout");
  printf("    -f <cfgfile>    read <cfgfile> as parameters\n\n");
  printf("    -h              displays this help text");
  printf("    -l <margin>     left margin in 1\"/600   (default: 150 = 0.25\")\n");
  printf("    -r <margin>     right margin in 1\"/600  (default: 150 = 0.25\")\n");
  printf("    -s <paper>      paper size (us,letter,legal,a4, default: us)\n");
  printf("    -t <margin>     top margin in 1\"/600    (default: 150 = 0.25\")\n");
  printf("    -v <model>      printer model (710, 712, 720, 722,  820, or 1000)\n");
  printf("    -x <offset>     vertical offset adjustment in 1\"/600\n");
  printf("    -y <offset>     horizontal offset adjustment in 1\"/600\n");
  printf("  The -x and -y options accumulate.  The -v option resets the horizontal and\n");
  printf("  vertical adjustments to an internal default.  <infile> and <outfile> default\n");
  printf("  to stdin and stdout.  '-' is a synonym for stdin and stdout.\n\n");
  printf("  Configuration files specified with the '-f' parameter have the following\n  format:\n\n");
  printf("    # Comment\n");
  printf("    <key1> <value1>\n");
  printf("    <key2> <value2>\n");
  printf("    [etc.]\n\n");
  printf("  Valid keys are 'version', 'xoffset', 'yoffset', 'topmargin', 'leftmargin',\n");
  printf("  'rightmargin', 'bottommargin', 'papersize', or any non-null truncated\n");
  printf("  version of these words.  Valid values are the same as with the corresponding\n");
  printf("  command-line parameters.  Parameters in the configuration file act as though\n");
  printf("  the corresponding parameters were substituted, in order, for the '-f'\n");
  printf("  parameter which specified the file.\n\n");
  printf("  The file /etc/pbm2ppa.conf, if it exists, is processed as a configuration\n");
  printf("  file before any command-line parameters are processed.\n\n");
}

void parm_version(char* arg)
{
  if(!strcasecmp(arg,"hp710") || !strcmp(arg,"710"))
    printer.version=HP720;
  else if(!strcasecmp(arg,"hp712") || !strcmp(arg,"712"))
    printer.version=HP720;
  else if(!strcasecmp(arg,"hp720") || !strcmp(arg,"720"))
    printer.version=HP720;
  else if(!strcasecmp(arg,"hp722") || !strcmp(arg,"722"))
    printer.version=HP720;
  else if(!strcasecmp(arg,"hp820") || !strcmp(arg,"820"))
    printer.version=HP820;
  else if(!strcasecmp(arg,"hp1000") || !strcmp(arg,"1000"))
    printer.version=HP1000;
  else
  {
    fprintf(stderr,"parm_version(): unknown printer version '%s'\n",arg);
    exit(1);
  }
  set_printer_specific_defaults();
}

void parm_iversion(int arg)
{
  switch(arg)
  {
  case 710:
  case 712:
  case 720:
  case 722:
    printer.version=HP720;
    break;
  case 820:
    printer.version=HP820;
    break;
  case 1000:
    printer.version=HP1000;
    break;
  default:
    {
      fprintf(stderr,"parm_iversion(): unknown printer version '%d'\n",arg);
      exit(1);
    }
  }
  set_printer_specific_defaults();
}

void dump_config()
{
  printf("version:  ");
  switch(printer.version)
  {
  case HP720:  printf("HP720\n");  break;
  case HP820:  printf("HP820\n");  break;
  case HP1000: printf("HP1000\n"); break;
  }
  printf("x-offset: %d\ny-offset: %d\nmargins:\n top:    %d\n"
	 " left:   %d\n right:  %d\n bottom: %d\n",printer.x_offset,
	 printer.y_offset,printer.top_margin,printer.left_margin,
	 printer.right_margin,printer.bottom_margin);
  exit(0);
}

void read_config_file(char* fname)
{
  FILE* cfgfile=fopen(fname,"r");
  char line[1024],key[14],buf[10];
  int len,value,lineno=1;

  if(!cfgfile)
  {
    fprintf (stderr, "read_config_file(): couldn't open file '%s'\n", fname);
    exit(1);
  }

  while(fgets(line,1024,cfgfile))
  {
    if(strchr(line,'#'))
      *strchr(line,'#')=0;
    switch(sscanf(line,"%13s%9s",key,buf))
    {
    case 2:
      value=atoi(buf);
      len=strlen(key);
      if(!strncmp(key,"version",len))
	parm_iversion(value);
      else if(!strncmp(key,"xoffset",len))
	printer.x_offset=value;
      else if(!strncmp(key,"yoffset",len))
	printer.y_offset=value;
      else if(!strncmp(key,"topmargin",len))
	printer.top_margin=value;
      else if(!strncmp(key,"leftmargin",len))
	printer.left_margin=value;
      else if(!strncmp(key,"rightmargin",len))
	printer.right_margin=value;
      else if(!strncmp(key,"bottommargin",len))
	printer.bottom_margin=value;
      else if(!strncmp(key,"papersize",len))
      {
	if(!strcmp(buf,"us") || !strcmp(buf,"letter"))
	{
	  Width = LETTERWIDTH;
	  Height = LETTERHEIGHT;
	}
	else if(!strcmp(buf,"legal"))
	{
	  Width = LEGALWIDTH;
	  Height = LEGALHEIGHT;
	}
	else if(!strcmp(buf,"a4"))
	{
	  Width = A4WIDTH;
	  Height = A4HEIGHT;
	}
	else
	{
	  fprintf(stderr,"read_config_file(): unknown paper size %s\n",buf);
	  exit(1);
	}
      }
      else if(!strcmp(key,"dump"))
	dump_config();
      else 
      {
	fprintf(stderr,"read_config_file(): unrecognized parameter '%s' (line %d)\n\n",key,lineno);
	show_usage("pbm2ppa");
	exit(1);
      }
    case EOF:
    case 0: break;
    default:
      fprintf(stderr,"read_config_file(): error parsing config file (line %d)\n",lineno);
      exit(1);
    }
    lineno++;
  }

  if(feof(cfgfile))
  {
    fclose(cfgfile);
    return;
  }

  fprintf(stderr,"read_config_file(): error parsing config file\n");
  exit(1);
}

char* defaultcfgfile="/etc/pbm2ppa.conf";
int main (int argc, char *argv[])
{
  int argn;
  int got_in=0, got_out=0, do_continue=1;
  FILE *in=stdin, *out=stdout;
  struct stat tmpstat;

  printer.version = DEFAULT_PRINTER;
  Width = LETTERWIDTH;
  Height = LETTERHEIGHT;
  set_printer_specific_defaults();

  if(!stat(defaultcfgfile,&tmpstat))
    read_config_file(defaultcfgfile);

  for(argn=1; argn<argc; argn++)
  {
    if(!strcmp(argv[argn],"-h"))
    {
      show_usage(*argv);
      return 0;
    }
    else if(!strcmp(argv[argn],"-d"))
      dump_config();
    else if(argn+1<argc)
    {
      do_continue=1;
      if(!strcmp(argv[argn],"-v"))
	parm_version(argv[++argn]);
      else if(!strcmp(argv[argn],"-x"))
	printer.x_offset+=atoi(argv[++argn]);
      else if(!strcmp(argv[argn],"-y"))
	printer.y_offset+=atoi(argv[++argn]);
      else if(!strcmp(argv[argn],"-t"))
	printer.top_margin=atoi(argv[++argn]);
      else if(!strcmp(argv[argn],"-l"))
	printer.left_margin=atoi(argv[++argn]);
      else if(!strcmp(argv[argn],"-r"))
	printer.right_margin=atoi(argv[++argn]);
      else if(!strcmp(argv[argn],"-b"))
	printer.bottom_margin=atoi(argv[++argn]);
      else if(!strcmp(argv[argn],"-s"))
      {
	argn++;
	if(!strcmp(argv[argn],"us") || !strcmp(argv[argn],"letter"))
	{
	  Width = LETTERWIDTH;
	  Height = LETTERHEIGHT;
	}
	else if(!strcmp(argv[argn],"legal"))
	{
	  Width = LEGALWIDTH;
	  Height = LEGALHEIGHT;
	}
	else if(!strcmp(argv[argn],"a4"))
	{
	  Width = A4WIDTH;
	  Height = A4HEIGHT;
	}
	else
	{
	  fprintf(stderr,"unknown paper size %s\n",argv[argn]);
	  return 1;
	}
      }
      else if(!strcmp(argv[argn],"-f"))
	read_config_file(argv[++argn]);
      else do_continue=0;
      if(do_continue) continue;
    }

    if(!got_in)
    {
      if (strcmp (argv[argn], "-") == 0)
	in = stdin;
      else if ((in = fopen (argv[argn], "rb")) == NULL)
      {
	fprintf (stderr, "main(): couldn't open file '%s'\n", argv[argn]);
	return 1;
      }
      got_in=1;
    }
    else if(!got_out)
    {
      if (strcmp (argv[argn], "-") == 0)
	out = stdout;
      else if ((out = fopen (argv[argn], "wb")) == NULL)
      {
	fprintf (stderr, "main(): couldn't open file '%s'\n", argv[argn]);
	return 1;
      }
      got_out=1;
    }
    else
    {
      fprintf(stderr,"main(): unrecognized parameter '%s'\n",argv[argn]);
      show_usage(*argv);
      return 1;
    }
  }

  Pwidth=(Width+7)/8;
  printer.fptr=out;

  return print_pbm (in);
}


