/*
 * Copyright (c) 2008 Google, Inc
 * Contributed by Stephane Eranian <eranian@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * This file is part of libpfm, a performance monitoring support library for
 * applications on Linux.
 */
static pme_nhm_entry_t corei7_pe[]={
	/*
	 * BEGIN architected events
	 */
	{.pme_name = "UNHALTED_CORE_CYCLES",
		.pme_code = 0x003c,
		.pme_cntmsk = 0x2000f,
		.pme_flags = PFMLIB_NHM_FIXED1,
		.pme_desc =  "count core clock cycles whenever the clock signal on the specific core is running (not halted). Alias to event CPU_CLK_UNHALTED:THREAD"
	},
	{.pme_name = "INSTRUCTION_RETIRED",
		.pme_code = 0x00c0,
		.pme_cntmsk = 0x1000f,
		.pme_flags = PFMLIB_NHM_FIXED0|PFMLIB_NHM_PEBS,
		.pme_desc =  "count the number of instructions at retirement. Alias to event INST_RETIRED:ANY_P",
	},
	{.pme_name = "INSTRUCTIONS_RETIRED",
		.pme_code = 0x00c0,
		.pme_cntmsk = 0x1000f,
		.pme_flags = PFMLIB_NHM_FIXED0|PFMLIB_NHM_PEBS,
		.pme_desc =  "This is an alias for INSTRUCTION_RETIRED",
	},
	{.pme_name = "UNHALTED_REFERENCE_CYCLES",
		.pme_code = 0x013c,
		.pme_cntmsk = 0x40000,
		.pme_flags = PFMLIB_NHM_FIXED2_ONLY,
		.pme_desc =  "Unhalted reference cycles",
	},
	{.pme_name = "LLC_REFERENCES",
		.pme_code = 0x4f2e,
		.pme_cntmsk = 0xf,
		.pme_desc =  "count each request originating from the core to reference a cache line in the last level cache. The count may include speculation, but excludes cache line fills due to hardware prefetch. Alias to L2_RQSTS:SELF_DEMAND_MESI",
	},
	{.pme_name = "LAST_LEVEL_CACHE_REFERENCES",
		.pme_code = 0x4f2e,
		.pme_cntmsk = 0xf,
		.pme_desc =  "This is an alias for LLC_REFERENCES",
	},
	{.pme_name = "LLC_MISSES",
		.pme_code = 0x412e,
		.pme_cntmsk = 0xf,
		.pme_desc =  "count each cache miss condition for references to the last level cache. The event count may include speculation, but excludes cache line fills due to hardware prefetch. Alias to event L2_RQSTS:SELF_DEMAND_I_STATE",
	},
	{.pme_name = "LAST_LEVEL_CACHE_MISSES",
		.pme_code = 0x412e,
		.pme_cntmsk = 0xf,
		.pme_desc =  "This is an alias for LLC_MISSES",
	},
	{.pme_name = "BRANCH_INSTRUCTIONS_RETIRED",
		.pme_code = 0x00c4,
		.pme_cntmsk = 0xf,
		.pme_desc =  "count branch instructions at retirement. Specifically, this event counts the retirement of the last micro-op of a branch instruction. Alias to event BR_INST_RETIRED:ANY",
	},
	/*
	 * BEGIN core specific events
	 */
	{ .pme_name   = "ARITH",
		.pme_desc   = "Counts arithmetic multiply and divide operations",
		.pme_code   = 0x14,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "CYCLES_DIV_BUSY",
				.pme_udesc  = "Counts the number of cycles the divider is busy executing divide or square root operations. The divide can be integer, X87 or Streaming SIMD Extensions (SSE). The square root operation can be either X87 or SSE.",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DIV",
				.pme_udesc  = "Counts the number of divide or square root operations. The divide can be integer, X87 or Streaming SIMD Extensions (SSE). The square root operation can be either X87 or SSE.",
				.pme_ucode  = 0x01 | (1<<16) | (1<<15) | (1<<10),    /* cmask=1 invert=1 edge=1 */
				.pme_uflags = 0,
			},
			{ .pme_uname  = "MUL",
				.pme_udesc  = "Counts the number of multiply operations executed. This includes integer as well as floating point multiply operations but excludes DPPS mul and MPSAD.",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 3
	},

	{ .pme_name   = "BACLEAR",
		.pme_desc   = "Branch address calculator",
		.pme_code   = 0xE6,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "BAD_TARGET",
				.pme_udesc  = "BACLEAR asserted with bad target address",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "CLEAR",
				.pme_udesc  = "BACLEAR asserted, regardless of cause",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 2
	},
	{ .pme_name   = "BACLEAR_FORCE_IQ",
		.pme_desc   = "Instruction queue forced BACLEAR",
		.pme_code   = 0x01A7,
		.pme_flags  = 0,
	},
	{ .pme_name   = "BOGUS_BR",
		.pme_desc   = "Counts the number of bogus branches.",
		.pme_code   = 0x01E4,
		.pme_flags  = 0,
	},
	{ .pme_name   = "BPU_CLEARS",
		.pme_desc   = "Branch prediction Unit clears",
		.pme_code   = 0xE8,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "EARLY",
				.pme_udesc  = "Early Branch Prediciton Unit clears",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LATE",
				.pme_udesc  = "Late Branch Prediction Unit clears",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "ANY",
				.pme_udesc  = "count any Branch Prediction Unit clears",
				.pme_ucode  = 0x03,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 3
	},
	{ .pme_name   = "BPU_MISSED_CALL_RET",
		.pme_desc   = "Branch prediction unit missed call or return",
		.pme_code   = 0x01E5,
		.pme_flags  = 0,
	},
	{ .pme_name   = "BR_INST_DECODED",
		.pme_desc   = "Branch instructions decoded",
		.pme_code   = 0x01E0,
		.pme_flags  = 0,
	},
	{ .pme_name   = "BR_INST_EXEC",
		.pme_desc   = "Branch instructions executed",
		.pme_code   = 0x88,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "Branch instructions executed",
				.pme_ucode  = 0x7F,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "COND",
				.pme_udesc  = "Conditional branch instructions executed",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DIRECT",
				.pme_udesc  = "Unconditional branches executed",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DIRECT_NEAR_CALL",
				.pme_udesc  = "Unconditional call branches executed",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "INDIRECT_NEAR_CALL",
				.pme_udesc  = "Indirect call branches executed",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "INDIRECT_NON_CALL",
				.pme_udesc  = "Indirect non call branches executed",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "NEAR_CALLS",
				.pme_udesc  = "Call branches executed",
				.pme_ucode  = 0x30,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "NON_CALLS",
				.pme_udesc  = "All non call branches executed",
				.pme_ucode  = 0x07,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RETURN_NEAR",
				.pme_udesc  = "Indirect return branches executed",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "TAKEN",
				.pme_udesc  = "Taken branches executed",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 10
	},
	{ .pme_name   = "BR_INST_RETIRED",
		.pme_desc   = "Retired branch instructions",
		.pme_code   = 0xC4,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ALL_BRANCHES",
				.pme_udesc  = "Retired branch instructions (Precise Event)",
				.pme_ucode  = 0x04,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "CONDITIONAL",
				.pme_udesc  = "Retired conditional branch instructions (Precise Event)",
				.pme_ucode  = 0x01,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "NEAR_CALL",
				.pme_udesc  = "Retired near call instructions (Precise Event)",
				.pme_ucode  = 0x02,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
		},
		.pme_numasks = 3
	},
	{ .pme_name   = "BR_MISP_EXEC",
		.pme_desc   = "Mispredicted branches executed",
		.pme_code   = 0x89,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "Mispredicted branches executed",
				.pme_ucode  = 0x7F,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "COND",
				.pme_udesc  = "Mispredicted conditional branches executed",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DIRECT",
				.pme_udesc  = "Mispredicted unconditional branches executed",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DIRECT_NEAR_CALL",
				.pme_udesc  = "Mispredicted non call branches executed",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "INDIRECT_NEAR_CALL",
				.pme_udesc  = "Mispredicted indirect call branches executed",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "INDIRECT_NON_CALL",
				.pme_udesc  = "Mispredicted indirect non call branches executed",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "NEAR_CALLS",
				.pme_udesc  = "Mispredicted call branches executed",
				.pme_ucode  = 0x30,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "NON_CALLS",
				.pme_udesc  = "Mispredicted non call branches executed",
				.pme_ucode  = 0x07,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RETURN_NEAR",
				.pme_udesc  = "Mispredicted return branches executed",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "TAKEN",
				.pme_udesc  = "Mispredicted taken branches executed",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 10
	},
	{ .pme_name   = "BR_MISP_RETIRED",
		.pme_desc   = "Count Mispredicted Branch Activity",
		.pme_code   = 0xC5,
		.pme_flags  = 0,
		.pme_umasks = {
			{ .pme_uname  = "NEAR_CALL",
				.pme_udesc  = "Counts mispredicted direct and indirect near unconditional retired calls",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 1
	},
	{ .pme_name   = "CACHE_LOCK_CYCLES",
		.pme_desc   = "Cache lock cycles",
		.pme_code   = 0x63,
		.pme_flags  = PFMLIB_NHM_PMC01,
		.pme_umasks = {
			{ .pme_uname  = "L1D",
				.pme_udesc  = "Cycles L1D locked",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "L1D_L2",
				.pme_udesc  = "Cycles L1D and L2 locked",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 2
	},
	{ .pme_name   = "CPU_CLK_UNHALTED",
		.pme_desc   = "Cycles when processor is not in halted state",
		.pme_code   = 0x3C,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "THREAD_P",
				.pme_udesc  = "Cycles when thread is not halted (programmable counter)",
				.pme_ucode  = 0x00,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "REF_P",
				.pme_udesc  = "Reference base clock (133 Mhz) cycles when thread is not halted",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 2
	},
	{ .pme_name   = "DTLB_LOAD_MISSES",
		.pme_desc   = "Data TLB load misses",
		.pme_code   = 0x08,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "DTLB load misses",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PDE_MISS",
				.pme_udesc  = "DTLB load miss caused by low part of address",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "WALK_COMPLETED",
				.pme_udesc  = "DTLB load miss page walks complete",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "STLB_HIT",
				.pme_udesc  = "DTLB second level hit",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PDP_MISS",
				.pme_udesc  = "Number of DTLB cache load misses where the high part of the linear to physical address translation was missed",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LARGE_WALK_COMPLETED",
				.pme_udesc  = "Counts number of completed large page walks due to load miss in the STLB",
				.pme_ucode  = 0x80,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 6
	},
	{ .pme_name   = "DTLB_MISSES",
		.pme_desc   = "Data TLB misses",
		.pme_code   = 0x49,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "DTLB misses",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "STLB_HIT",
				.pme_udesc  = "DTLB first level misses but second level hit",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "WALK_COMPLETED",
				.pme_udesc  = "DTLB miss page walks",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PDE_MISS",
				.pme_udesc  = "Number of DTLB cache misses where the low part of the linear to physical address translation was missed",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PDP_MISS",
				.pme_udesc  = "Number of DTLB misses where the high part of the linear to physical address translation was missed",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LARGE_WALK_COMPLETED",
				.pme_udesc  = "Counts number of completed large page walks due to misses in the STLB",
				.pme_ucode  = 0x80,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 6
	},
	{ .pme_name   = "EPT",
		.pme_desc   = "Extended Page Directory",
		.pme_code   = 0x4F,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "EPDE_MISS",
				.pme_udesc  = "Extended Page Directory Entry miss",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "EPDPE_MISS",
				.pme_udesc  = "Extended Page Directory Pointer miss",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "EPDPE_HIT",
				.pme_udesc  = "Extended Page Directory Pointer hit",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 3
	},
	{ .pme_name   = "ES_REG_RENAMES",
		.pme_desc   = "ES segment renames",
		.pme_code   = 0x01D5,
		.pme_flags  = 0,
	},
	{ .pme_name   = "FP_ASSIST",
		.pme_desc   = "Floating point assists",
		.pme_code   = 0xF7,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ALL",
				.pme_udesc  = "Floating point assists (Precise Event)",
				.pme_ucode  = 0x01,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "INPUT",
				.pme_udesc  = "Floating poiint assists for invalid input value (Precise Event)",
				.pme_ucode  = 0x04,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "OUTPUT",
				.pme_udesc  = "Floating point assists for invalid output value (Precise Event)",
				.pme_ucode  = 0x02,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
		},
		.pme_numasks = 3
	},
	{ .pme_name   = "FP_COMP_OPS_EXE",
		.pme_desc   = "Floating point computational micro-ops",
		.pme_code   = 0x10,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "MMX",
				.pme_udesc  = "MMX Uops",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "SSE_DOUBLE_PRECISION",
				.pme_udesc  = "SSE* FP double precision Uops",
				.pme_ucode  = 0x80,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "SSE_FP",
				.pme_udesc  = "SSE and SSE2 FP Uops",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "SSE_FP_PACKED",
				.pme_udesc  = "SSE FP packed Uops",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "SSE_FP_SCALAR",
				.pme_udesc  = "SSE FP scalar Uops",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "SSE_SINGLE_PRECISION",
				.pme_udesc  = "SSE* FP single precision Uops",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "SSE2_INTEGER",
				.pme_udesc  = "SSE2 integer Uops",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "X87",
				.pme_udesc  = "Computational floating-point operations executed",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 8
	},
	{ .pme_name   = "FP_MMX_TRANS",
		.pme_desc   = "Floating Point to and from MMX transitions",
		.pme_code   = 0xCC,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "All Floating Point to and from MMX transitions",
				.pme_ucode  = 0x03,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "TO_FP",
				.pme_udesc  = "Transitions from MMX to Floating Point instructions",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "TO_MMX",
				.pme_udesc  = "Transitions from Floating Point to MMX instructions",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 3
	},
	{ .pme_name   = "HW_INT",
		.pme_desc   = "Hardware interrupts",
		.pme_code   = 0x1D,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "RCV",
				.pme_udesc  = "Number of interrupt received",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "CYCLES_MASKED",
				.pme_udesc  = "Number of cycles interrupt are masked",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "CYCLES_PENDING_AND_MASKED",
				.pme_udesc  = "Number of cycles interrupts are pending and masked",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 3
	},
	{ .pme_name   = "IFU_IVC",
		.pme_desc   = "Instruction Fetch unit victim cache",
		.pme_code   = 0x81,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "FULL",
				.pme_udesc  = "Instruction Fetche unit victim cache full",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "L1I_EVICTION",
				.pme_udesc  = "L1 Instruction cache evictions",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 2
	},
	{ .pme_name   = "ILD_STALL",
		.pme_desc   = "Instruction Length Decoder stalls",
		.pme_code   = 0x87,
		.pme_flags  = 0,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "Any Instruction Length Decoder stall cycles",
				.pme_ucode  = 0x0F,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "IQ_FULL",
				.pme_udesc  = "Instruction Queue full stall cycles",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LCP",
				.pme_udesc  = "Length Change Prefix stall cycles",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "MRU",
				.pme_udesc  = "Stall cycles due to BPU MRU bypass",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "REGEN",
				.pme_udesc  = "Regen stall cycles",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 5
	},
	{ .pme_name   = "INST_DECODED",
		.pme_desc   = "Instructions decoded",
		.pme_code   = 0x18,
		.pme_flags  = 0,
		.pme_umasks = {
			{ .pme_uname  = "DEC0",
				.pme_udesc  = "Instructions that must be decoded by decoder 0",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 1
	},
	{ .pme_name   = "INST_QUEUE_WRITES",
		.pme_desc   = "Instructions written to instruction queue.",
		.pme_code   = 0x0117,
		.pme_flags  = 0,
	},
	{ .pme_name   = "INST_QUEUE_WRITE_CYCLES",
		.pme_desc   = "Cycles instructions are written to the instruction queue",
		.pme_code   = 0x011E,
		.pme_flags  = 0,
	},
	{ .pme_name   = "INST_RETIRED",
		.pme_desc   = "Instructions retired",
		.pme_code   = 0xC0,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY_P",
				.pme_udesc  = "Instructions Retired (Precise Event)",
				.pme_ucode  = 0x00,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "X87",
				.pme_udesc  = "Retired floating-point operations (Precise Event)",
				.pme_ucode  = 0x02,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
		},
		.pme_numasks = 2
	},
	{ .pme_name   = "IO_TRANSACTIONS",
		.pme_desc   = "I/O transactions",
		.pme_code   = 0x016C,
		.pme_flags  = 0,
	},
	{ .pme_name   = "ITLB_FLUSH",
		.pme_desc   = "Counts the number of ITLB flushes",
		.pme_code   = 0x01AE,
		.pme_flags  = 0,
	},
	{ .pme_name   = "ITLB_MISSES",
		.pme_desc   = "Instruction TLB misses",
		.pme_code   = 0x85,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "ITLB miss",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "WALK_COMPLETED",
				.pme_udesc  = "ITLB miss page walks",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "STLB_HIT",
				.pme_udesc  = "Counts the number of ITLB misses that hit in the second level TLB",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PDE_MISS",
				.pme_udesc  = "Number of ITLB misses where the low part of the linear to physical address translation was missed",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PDP_MISS",
				.pme_udesc  = "Number of ITLB misses where the high part of the linear to physical address translation was missed",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LARGE_WALK_COMPLETED",
				.pme_udesc  = "Counts number of completed large page walks due to misses in the STLB",
				.pme_ucode  = 0x80,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 6
	},
	{ .pme_name   = "ITLB_MISS_RETIRED",
		.pme_desc   = "Retired instructions that missed the ITLB (Precise Event)",
		.pme_code   = 0x20C8,
		.pme_flags  = PFMLIB_NHM_PEBS,
	},
	{ .pme_name   = "L1D",
		.pme_desc   = "L1D cache",
		.pme_code   = 0x51,
		.pme_flags  = PFMLIB_NHM_PMC01|PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "M_EVICT",
				.pme_udesc  = "L1D cache lines replaced in M state",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "M_REPL",
				.pme_udesc  = "L1D cache lines allocated in the M state",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "M_SNOOP_EVICT",
				.pme_udesc  = "L1D snoop eviction of cache lines in M state",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "REPL",
				.pme_udesc  = "L1 data cache lines allocated",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 4
	},
	{ .pme_name   = "L1D_ALL_REF",
		.pme_desc   = "L1D references",
		.pme_code   = 0x43,
		.pme_flags  = PFMLIB_NHM_PMC01|PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "All references to the L1 data cache",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "CACHEABLE",
				.pme_udesc  = "L1 data cacheable reads and writes",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 2
	},
	{ .pme_name   = "L1D_CACHE_LD",
		.pme_desc   = "L1D  cacheable loads. WARNING: event may overcount loads",
		.pme_code   = 0x40,
		.pme_flags  = PFMLIB_NHM_PMC01|PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "E_STATE",
				.pme_udesc  = "L1 data cache read in E state",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "I_STATE",
				.pme_udesc  = "L1 data cache read in I state (misses)",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "M_STATE",
				.pme_udesc  = "L1 data cache read in M state",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "MESI",
				.pme_udesc  = "L1 data cache reads",
				.pme_ucode  = 0x0F,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "S_STATE",
				.pme_udesc  = "L1 data cache read in S state",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 5
	},
	{ .pme_name   = "L1D_CACHE_LOCK",
		.pme_desc   = "L1 data cache load lock",
		.pme_code   = 0x42,
		.pme_flags  = PFMLIB_NHM_PMC01|PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "E_STATE",
				.pme_udesc  = "L1 data cache load locks in E state",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "HIT",
				.pme_udesc  = "L1 data cache load lock hits. WARNING: overcounts by 3x",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "M_STATE",
				.pme_udesc  = "L1 data cache load locks in M state. WARNING: overcounts by 3x",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "S_STATE",
				.pme_udesc  = "L1 data cache load locks in S state",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 4
	},
	{ .pme_name   = "L1D_CACHE_LOCK_FB_HIT",
		.pme_desc   = "L1D load lock accepted in fill buffer",
		.pme_code   = 0x0153,
		.pme_flags  = PFMLIB_NHM_PMC01,
	},
	{ .pme_name   = "L1D_CACHE_PREFETCH_LOCK_FB_HIT",
		.pme_desc   = "L1D prefetch load lock accepted in fill buffer",
		.pme_code   = 0x0152,
		.pme_flags  = PFMLIB_NHM_PMC01,
	},
	{ .pme_name   = "L1D_CACHE_ST",
		.pme_desc   = "L1 data cache stores",
		.pme_code   = 0x41,
		.pme_flags  = PFMLIB_NHM_PMC01|PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "E_STATE",
				.pme_udesc  = "L1 data cache stores in E state",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "I_STATE",
				.pme_udesc  = "L1 data cache store in the I state",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "M_STATE",
				.pme_udesc  = "L1 data cache stores in M state",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "S_STATE",
				.pme_udesc  = "L1 data cache stores in S state",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "MESI",
				.pme_udesc  = "L1 data cache store in all states",
				.pme_ucode  = 0x0F,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 5
	},
	{ .pme_name   = "L1D_PREFETCH",
		.pme_desc   = "L1D hardware prefetch",
		.pme_code   = 0x4E,
		.pme_flags  = PFMLIB_NHM_PMC01|PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "MISS",
				.pme_udesc  = "L1D hardware prefetch misses",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "REQUESTS",
				.pme_udesc  = "L1D hardware prefetch requests",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "TRIGGERS",
				.pme_udesc  = "L1D hardware prefetch requests triggered",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 3
	},
	{ .pme_name   = "L1D_WB_L2",
		.pme_desc   = "L1 writebacks to L2",
		.pme_code   = 0x28,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "E_STATE",
				.pme_udesc  = "L1 writebacks to L2 in E state",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "I_STATE",
				.pme_udesc  = "L1 writebacks to L2 in I state (misses)",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "M_STATE",
				.pme_udesc  = "L1 writebacks to L2 in M state",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "S_STATE",
				.pme_udesc  = "L1 writebacks to L2 in S state",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "MESI",
				.pme_udesc  = "All L1 writebacks to L2",
				.pme_ucode  = 0x0F,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 5
	},
	{ .pme_name   = "L1I",
		.pme_desc   = "L1I instruction fetches",
		.pme_code   = 0x80,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "CYCLES_STALLED",
				.pme_udesc  = "L1I instruction fetch stall cycles",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "HITS",
				.pme_udesc  = "L1I instruction fetch hits",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "MISSES",
				.pme_udesc  = "L1I instruction fetch misses",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "READS",
				.pme_udesc  = "L1I Instruction fetches",
				.pme_ucode  = 0x03,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 4
	},
	{ .pme_name   = "L1I_OPPORTUNISTIC_HITS",
		.pme_desc   = "Opportunistic hits in streaming",
		.pme_code   = 0x0183,
		.pme_flags  = 0,
	},
	{ .pme_name   = "L2_DATA_RQSTS",
		.pme_desc   = "L2 data requests",
		.pme_code   = 0x26,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "All L2 data requests",
				.pme_ucode  = 0xFF,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DEMAND_E_STATE",
				.pme_udesc  = "L2 data demand loads in E state",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DEMAND_I_STATE",
				.pme_udesc  = "L2 data demand loads in I state (misses)",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DEMAND_M_STATE",
				.pme_udesc  = "L2 data demand loads in M state",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DEMAND_MESI",
				.pme_udesc  = "L2 data demand requests",
				.pme_ucode  = 0x0F,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DEMAND_S_STATE",
				.pme_udesc  = "L2 data demand loads in S state",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PREFETCH_E_STATE",
				.pme_udesc  = "L2 data prefetches in E state",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PREFETCH_I_STATE",
				.pme_udesc  = "L2 data prefetches in the I state (misses)",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PREFETCH_M_STATE",
				.pme_udesc  = "L2 data prefetches in M state",
				.pme_ucode  = 0x80,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PREFETCH_MESI",
				.pme_udesc  = "All L2 data prefetches",
				.pme_ucode  = 0xF0,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PREFETCH_S_STATE",
				.pme_udesc  = "L2 data prefetches in the S state",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 11
	},
	{ .pme_name   = "L2_HW_PREFETCH",
		.pme_desc   = "L2 HW prefetches",
		.pme_code   = 0xF3,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "HIT",
				.pme_udesc  = "Count L2 HW prefetcher detector hits",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "ALLOC",
				.pme_udesc  = "Count L2 HW prefetcher allocations",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DATA_TRIGGER",
				.pme_udesc  = "Count L2 HW data prefetcher triggered",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "CODE_TRIGGER",
				.pme_udesc  = "Count L2 HW code prefetcher triggered",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DCA_TRIGGER",
				.pme_udesc  = "Count L2 HW DCA prefetcher triggered",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "KICK_START",
				.pme_udesc  = "Count L2 HW prefetcher kick started",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 6
	},
	{ .pme_name   = "L2_LINES_IN",
		.pme_desc   = "L2 lines allocated",
		.pme_code   = 0xF1,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "any L2 lines allocated",
				.pme_ucode  = 0x07,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "E_STATE",
				.pme_udesc  = "L2 lines allocated in the E state",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "S_STATE",
				.pme_udesc  = "L2 lines allocated in the S state",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 3
	},
	{ .pme_name   = "L2_LINES_OUT",
		.pme_desc   = "L2 lines evicted",
		.pme_code   = 0xF2,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "L2 lines evicted",
				.pme_ucode  = 0x0F,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DEMAND_CLEAN",
				.pme_udesc  = "L2 lines evicted by a demand request",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DEMAND_DIRTY",
				.pme_udesc  = "L2 modified lines evicted by a demand request",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PREFETCH_CLEAN",
				.pme_udesc  = "L2 lines evicted by a prefetch request",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PREFETCH_DIRTY",
				.pme_udesc  = "L2 modified lines evicted by a prefetch request",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 5
	},
	{ .pme_name   = "L2_RQSTS",
		.pme_desc   = "L2 requests",
		.pme_code   = 0x24,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "MISS",
				.pme_udesc  = "All L2 misses",
				.pme_ucode  = 0xAA,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "REFERENCES",
				.pme_udesc  = "All L2 requests",
				.pme_ucode  = 0xFF,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "IFETCH_HIT",
				.pme_udesc  = "L2 instruction fetch hits",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "IFETCH_MISS",
				.pme_udesc  = "L2 instruction fetch misses",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "IFETCHES",
				.pme_udesc  = "L2 instruction fetches",
				.pme_ucode  = 0x30,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LD_HIT",
				.pme_udesc  = "L2 load hits",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LD_MISS",
				.pme_udesc  = "L2 load misses",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LOADS",
				.pme_udesc  = "L2 requests",
				.pme_ucode  = 0x03,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PREFETCH_HIT",
				.pme_udesc  = "L2 prefetch hits",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PREFETCH_MISS",
				.pme_udesc  = "L2 prefetch misses",
				.pme_ucode  = 0x80,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PREFETCHES",
				.pme_udesc  = "All L2 prefetches",
				.pme_ucode  = 0xC0,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RFO_HIT",
				.pme_udesc  = "L2 RFO hits",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RFO_MISS",
				.pme_udesc  = "L2 RFO misses",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RFOS",
				.pme_udesc  = "L2 RFO requests",
				.pme_ucode  = 0x0C,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 14
	},
	{ .pme_name   = "L2_TRANSACTIONS",
		.pme_desc   = "L2 transactions",
		.pme_code   = 0xF0,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "All L2 transactions",
				.pme_ucode  = 0x80,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "FILL",
				.pme_udesc  = "L2 fill transactions",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "IFETCH",
				.pme_udesc  = "L2 instruction fetch transactions",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "L1D_WB",
				.pme_udesc  = "L1D writeback to L2 transactions",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LOAD",
				.pme_udesc  = "L2 Load transactions",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PREFETCH",
				.pme_udesc  = "L2 prefetch transactions",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RFO",
				.pme_udesc  = "L2 RFO transactions",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "WB",
				.pme_udesc  = "L2 writeback to LLC transactions",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 8
	},
	{ .pme_name   = "L2_WRITE",
		.pme_desc   = "L2 demand lock/store RFO",
		.pme_code   = 0x27,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "LOCK_E_STATE",
				.pme_udesc  = "L2 demand lock RFOs in E state",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LOCK_I_STATE",
				.pme_udesc  = "L2 demand lock RFOs in I state (misses)",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LOCK_S_STATE",
				.pme_udesc  = "L2 demand lock RFOs in S state",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LOCK_HIT",
				.pme_udesc  = "All demand L2 lock RFOs that hit the cache",
				.pme_ucode  = 0xE0,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LOCK_M_STATE",
				.pme_udesc  = "L2 demand lock RFOs in M state",
				.pme_ucode  = 0x80,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LOCK_MESI",
				.pme_udesc  = "All demand L2 lock RFOs",
				.pme_ucode  = 0xF0,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RFO_HIT",
				.pme_udesc  = "All L2 demand store RFOs that hit the cache",
				.pme_ucode  = 0x0E,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RFO_E_STATE",
				.pme_udesc  = "L2 demand store RFOs in the E state (exclusive)",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RFO_I_STATE",
				.pme_udesc  = "L2 demand store RFOs in I state (misses)",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RFO_M_STATE",
				.pme_udesc  = "L2 demand store RFOs in M state",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RFO_MESI",
				.pme_udesc  = "All L2 demand store RFOs",
				.pme_ucode  = 0x0F,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RFO_S_STATE",
				.pme_udesc  = "L2 demand store RFOs in S state",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 12
	},
	{ .pme_name   = "LARGE_ITLB",
		.pme_desc   = "Large instruction TLB",
		.pme_code   = 0x82,
		.pme_flags  = 0,
		.pme_umasks = {
			{ .pme_uname  = "HIT",
				.pme_udesc  = "Large ITLB hit",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 1
	},
	{ .pme_name   = "LOAD_DISPATCH",
		.pme_desc   = "Loads dispatched",
		.pme_code   = 0x13,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "All loads dispatched",
				.pme_ucode  = 0x07,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "MOB",
				.pme_udesc  = "Loads dispatched from the MOB",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RS",
				.pme_udesc  = "Loads dispatched that bypass the MOB",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RS_DELAYED",
				.pme_udesc  = "Loads dispatched from stage 305",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 4
	},
	{ .pme_name   = "LOAD_HIT_PRE",
		.pme_desc   = "Load operations conflicting with software prefetches",
		.pme_code   = 0x014C,
		.pme_flags  = PFMLIB_NHM_PMC01,
	},
	{ .pme_name   = "LONGEST_LAT_CACHE",
		.pme_desc   = "Longest latency cache reference",
		.pme_code   = 0x2E,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "REFERENCE",
				.pme_udesc  = "Longest latency cache reference",
				.pme_ucode  = 0x4F,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "MISS",
				.pme_udesc  = "Longest latency cache miss",
				.pme_ucode  = 0x41,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 2
	},
	{ .pme_name   = "LSD",
		.pme_desc   = "Loop stream detector",
		.pme_code   = 0xA8,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ACTIVE",
				.pme_udesc  = "Cycles when uops were delivered by the LSD",
				.pme_ucode  = 0x01 | (1<<16),
				.pme_uflags = 0,
			},
			{ .pme_uname  = "INACTIVE",
				.pme_udesc  = "Cycles no uops were delivered by the LSD",
				.pme_ucode  = 0x01 | (1<<16)|(1<<15),
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 2
	},
	{ .pme_name   = "MACHINE_CLEARS",
		.pme_desc   = "Machine Clear",
		.pme_code   = 0xC3,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "SMC",
				.pme_udesc  = "Self-Modifying Code detected",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "CYCLES",
				.pme_udesc  = "Cycles machine clear asserted",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "MEM_ORDER",
				.pme_udesc  = "Execution pipeline restart due to Memory ordering conflicts",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "FUSION_ASSIST",
				.pme_udesc  = "Counts the number of macro-fusion assists",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 4
	},
	{ .pme_name   = "MACRO_INSTS",
		.pme_desc   = "Macro-fused instructions",
		.pme_code   = 0xD0,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "DECODED",
				.pme_udesc  = "Instructions decoded",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "FUSIONS_DECODED",
				.pme_udesc  = "Macro-fused instructions decoded",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 2
	},
	{ .pme_name   = "MEMORY_DISAMBIGUATION",
		.pme_desc   = "Memory Disambiguation Activity",
		.pme_code   = 0x09,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "RESET",
				.pme_udesc  = "Counts memory disambiguation reset cycles",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "WATCHDOG",
				.pme_udesc  = "Counts the number of times the memory disambiguation watchdog kicked in",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "WATCH_CYCLES",
				.pme_udesc  = "Counts the cycles that the memory disambiguation watchdog is active",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 3
	},
	{ .pme_name   = "MEM_INST_RETIRED",
		.pme_desc   = "Memory instructions retired",
		.pme_code   = 0x0B,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "LATENCY_ABOVE_THRESHOLD",
				.pme_udesc  = "Memory instructions retired above programmed clocks, minimum value threhold is 4, requires PEBS",
				.pme_ucode  = 0x10,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "LOADS",
				.pme_udesc  = "Instructions retired which contains a load (Precise Event)",
				.pme_ucode  = 0x01,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "STORES",
				.pme_udesc  = "Instructions retired which contains a store (Precise Event)",
				.pme_ucode  = 0x02,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
		},
		.pme_numasks = 3
	},
	{ .pme_name   = "MEM_LOAD_RETIRED",
		.pme_desc   = "Retired loads",
		.pme_code   = 0xCB,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "DTLB_MISS",
				.pme_udesc  = "Retired loads that miss the DTLB (Precise Event)",
				.pme_ucode  = 0x80,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "HIT_LFB",
				.pme_udesc  = "Retired loads that miss L1D and hit an previously allocated LFB (Precise Event)",
				.pme_ucode  = 0x40,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "L1D_HIT",
				.pme_udesc  = "Retired loads that hit the L1 data cache (Precise Event)",
				.pme_ucode  = 0x01,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "L2_HIT",
				.pme_udesc  = "Retired loads that hit the L2 cache (Precise Event)",
				.pme_ucode  = 0x02,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "L3_MISS",
				.pme_udesc  = "Retired loads that miss the LLC cache (Precise Event)",
				.pme_ucode  = 0x10,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "LLC_MISS",
				.pme_udesc  = "This is an alias for L3_MISS",
				.pme_ucode  = 0x10,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "L3_UNSHARED_HIT",
				.pme_udesc  = "Retired loads that hit valid versions in the LLC cache (Precise Event)",
				.pme_ucode  = 0x04,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "LLC_UNSHARED_HIT",
				.pme_udesc  = "This is an alias for L3_UNSHARED_HIT",
				.pme_ucode  = 0x04,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "OTHER_CORE_L2_HIT_HITM",
				.pme_udesc  = "Retired loads that hit sibling core's L2 in modified or unmodified states (Precise Event)",
				.pme_ucode  = 0x08,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
		},
		.pme_numasks = 9
	},
	{ .pme_name   = "MEM_STORE_RETIRED",
		.pme_desc   = "Retired stores",
		.pme_code   = 0x0C,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "DTLB_MISS",
				.pme_udesc  = "Retired stores that miss the DTLB (Precise Event)",
				.pme_ucode  = 0x01,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
		},
		.pme_numasks = 1
	},
	{ .pme_name   = "MEM_UNCORE_RETIRED",
		.pme_desc   = "Load instructions retired which hit offcore",
		.pme_code   = 0x0F,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "OTHER_CORE_L2_HITM",
				.pme_udesc  = "Load instructions retired that HIT modified data in sibling core (Precise Event)",
				.pme_ucode  = 0x02,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "REMOTE_CACHE_LOCAL_HOME_HIT",
				.pme_udesc  = "Load instructions retired remote cache HIT data source (Precise Event)",
				.pme_ucode  = 0x08,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "REMOTE_DRAM",
				.pme_udesc  = "Load instructions retired remote DRAM and remote home-remote cache HITM (Precise Event)",
				.pme_ucode  = 0x10,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "LOCAL_DRAM",
				.pme_udesc  = "Load instructions retired with a data source of local DRAM or locally homed remote hitm (Precise Event)",
				.pme_ucode  = 0x20,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			/* Model 46 only (must be after common umasks) */
			{ .pme_uname  = "L3_DATA_MISS_UNKNOWN",
				.pme_udesc  = "Load instructions retired where the memory reference missed L3 and data source is unknown (Model 46 only, Precise Event)",
				.pme_ucode  = 0x01,
				.pme_umodel = 46,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			/* Model 46 only (must be after common umasks) */
			{ .pme_uname  = "UNCACHEABLE",
				.pme_udesc  = "Load instructions retired where the memory reference missed L1, L2, L3 caches and to perform I/O (Model 46 only, Precise Event)",
				.pme_ucode  = 0x80,
				.pme_umodel = 46,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
		},
		.pme_numasks = 6 /* patched at runtime for model 46 */
	},
	{ .pme_name   = "OFFCORE_REQUESTS",
		.pme_desc   = "Offcore memory requests",
		.pme_code   = 0xB0,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "All offcore requests",
				.pme_ucode  = 0x80,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "ANY_READ",
				.pme_udesc  = "Offcore read requests",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "ANY_RFO",
				.pme_udesc  = "Offcore RFO requests",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DEMAND_READ_CODE",
				.pme_udesc  = "Counts number of offcore demand code read requests. Does not count L2 prefetch requests.",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DEMAND_READ_DATA",
				.pme_udesc  = "Offcore demand data read requests",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DEMAND_RFO",
				.pme_udesc  = "Offcore demand RFO requests",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "L1D_WRITEBACK",
				.pme_udesc  = "Offcore L1 data cache writebacks",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "UNCACHED_MEM",
				.pme_udesc  = "Counts number of offcore uncached memory requests",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 8
	},
	{ .pme_name   = "OFFCORE_REQUESTS_SQ_FULL",
		.pme_desc   = "Counts cycles the Offcore Request buffer or Super Queue is full.",
		.pme_code   = 0x01B2,
		.pme_flags  = 0,
	},
	{ .pme_name   = "PARTIAL_ADDRESS_ALIAS",
		.pme_desc   = "False dependencies due to partial address aliasing",
		.pme_code   = 0x0107,
		.pme_flags  = 0,
	},
	{ .pme_name   = "PIC_ACCESSES",
		.pme_desc   = "Programmable interrupt controller",
		.pme_code   = 0xBA,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "TPR_READS",
				.pme_udesc  = "Counts number of TPR reads",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "TPR_WRITES",
				.pme_udesc  = "Counts number of TPR writes",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 2
	},
	{ .pme_name   = "RAT_STALLS",
		.pme_desc   = "Register allocation table stalls",
		.pme_code   = 0xD2,
		.pme_flags  = 0,
		.pme_umasks = {
			{ .pme_uname  = "FLAGS",
				.pme_udesc  = "Flag stall cycles",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "REGISTERS",
				.pme_udesc  = "Partial register stall cycles",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "ROB_READ_PORT",
				.pme_udesc  = "ROB read port stalls cycles",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "SCOREBOARD",
				.pme_udesc  = "Scoreboard stall cycles",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "ANY",
				.pme_udesc  = "All RAT stall cycles",
				.pme_ucode  = 0x0F,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 5
	},
	{ .pme_name   = "RESOURCE_STALLS",
		.pme_desc   = "Processor stalls",
		.pme_code   = 0xA2,
		.pme_flags  = 0,
		.pme_umasks = {
			{ .pme_uname  = "FPCW",
				.pme_udesc  = "FPU control word write stall cycles",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LOAD",
				.pme_udesc  = "Load buffer stall cycles",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "MXCSR",
				.pme_udesc  = "MXCSR rename stall cycles",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "RS_FULL",
				.pme_udesc  = "Reservation Station full stall cycles",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "STORE",
				.pme_udesc  = "Store buffer stall cycles",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "OTHER",
				.pme_udesc  = "Other Resource related stall cycles",
				.pme_ucode  = 0x80,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "ROB_FULL",
				.pme_udesc  = "ROB full stall cycles",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "ANY",
				.pme_udesc  = "Resource related stall cycles",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 8
	},
	{ .pme_name   = "SEG_RENAME_STALLS",
		.pme_desc   = "Segment rename stall cycles",
		.pme_code   = 0x01D4,
		.pme_flags  = 0,
	},
	{ .pme_name   = "SEGMENT_REG_LOADS",
		.pme_desc   = "Counts number of segment register loads",
		.pme_code   = 0x01F8,
		.pme_flags  = 0,
	},
	{ .pme_name   = "SIMD_INT_128",
		.pme_desc   = "128 bit SIMD integer operations",
		.pme_code   = 0x12,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "PACK",
				.pme_udesc  = "128 bit SIMD integer pack operations",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PACKED_ARITH",
				.pme_udesc  = "128 bit SIMD integer arithmetic operations",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PACKED_LOGICAL",
				.pme_udesc  = "128 bit SIMD integer logical operations",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PACKED_MPY",
				.pme_udesc  = "128 bit SIMD integer multiply operations",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PACKED_SHIFT",
				.pme_udesc  = "128 bit SIMD integer shift operations",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "SHUFFLE_MOVE",
				.pme_udesc  = "128 bit SIMD integer shuffle/move operations",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "UNPACK",
				.pme_udesc  = "128 bit SIMD integer unpack operations",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 7
	},
	{ .pme_name   = "SIMD_INT_64",
		.pme_desc   = "64 bit SIMD integer operations",
		.pme_code   = 0xFD,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "PACK",
				.pme_udesc  = "SIMD integer 64 bit pack operations",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PACKED_ARITH",
				.pme_udesc  = "SIMD integer 64 bit arithmetic operations",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PACKED_LOGICAL",
				.pme_udesc  = "SIMD integer 64 bit logical operations",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PACKED_MPY",
				.pme_udesc  = "SIMD integer 64 bit packed multiply operations",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PACKED_SHIFT",
				.pme_udesc  = "SIMD integer 64 bit shift operations",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "SHUFFLE_MOVE",
				.pme_udesc  = "SIMD integer 64 bit shuffle/move operations",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "UNPACK",
				.pme_udesc  = "SIMD integer 64 bit unpack operations",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 7
	},
	{ .pme_name   = "SNOOP_RESPONSE",
		.pme_desc   = "Snoop",
		.pme_code   = 0xB8,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "HIT",
				.pme_udesc  = "Thread responded HIT to snoop",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "HITE",
				.pme_udesc  = "Thread responded HITE to snoop",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "HITM",
				.pme_udesc  = "Thread responded HITM to snoop",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 3
	},
	{ .pme_name   = "SQ_FULL_STALL_CYCLES",
		.pme_desc   = "Counts cycles the Offcore Request buffer or Super Queue is full and request(s) are outstanding.",
		.pme_code   = 0x01F6,
		.pme_flags  = 0,
	},
	{ .pme_name   = "SQ_MISC",
		.pme_desc   = "Super Queue Activity Related to L2 Cache Access",
		.pme_code   = 0xF4,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "PROMOTION",
				.pme_udesc  = "Counts the number of L2 secondary misses that hit the Super Queue",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PROMOTION_POST_GO",
				.pme_udesc  = "Counts the number of L2 secondary misses during the Super Queue filling L2",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LRU_HINTS",
				.pme_udesc  = "Counts number of Super Queue LRU hints sent to L3",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "FILL_DROPPED",
				.pme_udesc  = "Counts the number of SQ L2 fills dropped due to L2 busy",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "SPLIT_LOCK",
				.pme_udesc  = "Super Queue lock splits across a cache line",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 5
	},
	{ .pme_name   = "SSE_MEM_EXEC",
		.pme_desc   = "Streaming SIMD executed",
		.pme_code   = 0x4B,
		.pme_flags  = 0,
		.pme_umasks = {
			{ .pme_uname  = "NTA",
				.pme_udesc  = "Streaming SIMD L1D NTA prefetch miss",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 1
	},
	{ .pme_name   = "SSEX_UOPS_RETIRED",
		.pme_desc   = "SIMD micro-ops retired",
		.pme_code   = 0xC7,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "PACKED_DOUBLE",
				.pme_udesc  = "SIMD Packed-Double Uops retired (Precise Event)",
				.pme_ucode  = 0x04,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "PACKED_SINGLE",
				.pme_udesc  = "SIMD Packed-Single Uops retired (Precise Event)",
				.pme_ucode  = 0x01,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "SCALAR_DOUBLE",
				.pme_udesc  = "SIMD Scalar-Double Uops retired (Precise Event)",
				.pme_ucode  = 0x08,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "SCALAR_SINGLE",
				.pme_udesc  = "SIMD Scalar-Single Uops retired (Precise Event)",
				.pme_ucode  = 0x02,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "VECTOR_INTEGER",
				.pme_udesc  = "SIMD Vector Integer Uops retired (Precise Event)",
				.pme_ucode  = 0x10,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
		},
		.pme_numasks = 5
	},
	{ .pme_name   = "STORE_BLOCKS",
		.pme_desc   = "Delayed loads",
		.pme_code   = 0x06,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "AT_RET",
				.pme_udesc  = "Loads delayed with at-Retirement block code",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "L1D_BLOCK",
				.pme_udesc  = "Cacheable loads delayed with L1D block code",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "NOT_STA",
				.pme_udesc  = "Loads delayed due to a store blocked for unknown data",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "STA",
				.pme_udesc  = "Loads delayed due to a store blocked for an unknown address",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 4
	},
	{ .pme_name   = "TWO_UOP_INSTS_DECODED",
		.pme_desc   = "Two micro-ops instructions decoded",
		.pme_code   = 0x0119,
		.pme_flags  = 0,
	},
	{ .pme_name   = "UOPS_DECODED_DEC0",
		.pme_desc   = "Micro-ops decoded by decoder 0",
		.pme_code   = 0x013D,
		.pme_flags  = 0,
	},
	{ .pme_name   = "UOPS_DECODED",
		.pme_desc   = "Micro-ops decoded",
		.pme_code   = 0xD1,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ESP_FOLDING",
				.pme_udesc  = "Stack pointer instructions decoded",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "ESP_SYNC",
				.pme_udesc  = "Stack pointer sync operations",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "MS",
				.pme_udesc  = "Uops decoded by Microcode Sequencer",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "MS_CYCLES_ACTIVE",
				.pme_udesc  = "cycles in which at least one uop is decoded by Microcode Sequencer",
				.pme_ucode  = 0x2 | (1<< 16), /* counter-mask = 1 */
			},

		},
		.pme_numasks = 4
	},
	{ .pme_name   = "UOPS_EXECUTED",
		.pme_desc   = "Micro-ops executed",
		.pme_code   = 0xB1,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "PORT0",
				.pme_udesc  = "Uops executed on port 0",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PORT1",
				.pme_udesc  = "Uops executed on port 1",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PORT2_CORE",
				.pme_udesc  = "Uops executed on port 2 (core count only)",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PORT3_CORE",
				.pme_udesc  = "Uops executed on port 3 (core count only)",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PORT4_CORE",
				.pme_udesc  = "Uops executed on port 4 (core count only)",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PORT5",
				.pme_udesc  = "Uops executed on port 5",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PORT015",
				.pme_udesc  = "Uops issued on ports 0, 1 or 5",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PORT234_CORE",
				.pme_udesc  = "Uops issued on ports 2, 3 or 4 (core count only)",
				.pme_ucode  = 0x80,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PORT015_STALL_CYCLES",
				.pme_udesc  = "Cycles no Uops issued on ports 0, 1 or 5",
				.pme_ucode  = 0x40 | (1<<16) | (1<<15), /* counter-mask=1, inv=1 */
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 9
	},
	{ .pme_name   = "UOPS_ISSUED",
		.pme_desc   = "Micro-ops issued",
		.pme_code   = 0x0E,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "Uops issued",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "STALLED_CYCLES",
				.pme_udesc  = "Cycles stalled no issued uops",
				.pme_ucode  = 0x01 | (1<<16) | (1<<15), /* counter-mask=1, inv=1 */
				.pme_uflags = 0,
			},
			{ .pme_uname  = "FUSED",
				.pme_udesc  = "Fused Uops issued",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 3
	},
	{ .pme_name   = "UOPS_RETIRED",
		.pme_desc   = "Micro-ops retired",
		.pme_code   = 0xC2,
		.pme_flags  = PFMLIB_NHM_UMASK_NCOMBO,
		.pme_umasks = {
			{ .pme_uname  = "ANY",
				.pme_udesc  = "Uops retired (Precise Event)",
				.pme_ucode  = 0x01,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "RETIRE_SLOTS",
				.pme_udesc  = "Retirement slots used (Precise Event)",
				.pme_ucode  = 0x02,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "ACTIVE_CYCLES",
				.pme_udesc  = "Cycles Uops are being retired (Precise Event)",
				.pme_ucode  = 0x01 | (1<< 16), /* counter mask = 1 */
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "STALL_CYCLES",
				.pme_udesc  = "Cycles No Uops retired (Precise Event)",
				.pme_ucode  = 0x01 | (1<<16) | (1<<15), /* counter-mask=1, inv=1 */
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
			{ .pme_uname  = "MACRO_FUSED",
				.pme_udesc  = "Macro-fused Uops retired (Precise Event)",
				.pme_ucode  = 0x04,
				.pme_uflags = PFMLIB_NHM_PEBS,
			},
		},
		.pme_numasks = 5
	},
	{ .pme_name   = "UOP_UNFUSION",
		.pme_desc   = "Micro-ops unfusions due to FP exceptions",
		.pme_code   = 0x01DB,
		.pme_flags  = 0,
	},
	/*
	 * BEGIN OFFCORE_RESPONSE
	 */
	{ .pme_name   = "OFFCORE_RESPONSE_0",
		.pme_desc   = "Offcore response",
		.pme_code   = 0x01B7,
		.pme_flags  = PFMLIB_NHM_OFFCORE_RSP0,
		.pme_umasks = {
			{ .pme_uname  = "DMND_DATA_RD",
				.pme_udesc  = "Request. Counts the number of demand and DCU prefetch data reads of full and partial cachelines as well as demand data page table entry cacheline reads. Does not count L2 data read prefetches or instruction fetches",
				.pme_ucode  = 0x01,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DMND_RFO",
				.pme_udesc  = "Request. Counts the number of demand and DCU prefetch reads for ownership (RFO) requests generated by a write to data cacheline. Does not count L2 RFO",
				.pme_ucode  = 0x02,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "DMND_IFETCH",
				.pme_udesc  = "Request. Counts the number of demand and DCU prefetch instruction cacheline reads. Does not count L2 code read prefetches",
				.pme_ucode  = 0x04,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "WB",
				.pme_udesc  = "Request. Counts the number of writeback (modified to exclusive) transactions",
				.pme_ucode  = 0x08,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PF_DATA_RD",
				.pme_udesc  = "Request. Counts the number of data cacheline reads generated by L2 prefetchers",
				.pme_ucode  = 0x10,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PF_RFO",
				.pme_udesc  = "Request. Counts the number of RFO requests generated by L2 prefetchers",
				.pme_ucode  = 0x20,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "PF_IFETCH",
				.pme_udesc  = "Request. Counts the number of code reads generated by L2 prefetchers",
				.pme_ucode  = 0x40,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "OTHER",
				.pme_udesc  = "Request. Counts one of the following transaction types, including L3 invalidate, I/O, full or partial writes, WC or non-temporal stores, CLFLUSH, Fences, lock, unlock, split lock",
				.pme_ucode  = 0x80,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "ANY_REQUEST",
				.pme_udesc  = "Request. Counts any request type",
				.pme_ucode  = 0xff,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "UNCORE_HIT",
				.pme_udesc  = "Response. Counts L3 Hit: local or remote home requests that hit L3 cache in the uncore with no coherency actions required (snooping)",
				.pme_ucode  = 0x100,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "OTHER_CORE_HIT_SNP",
				.pme_udesc  = "Response. Counts L3 Hit: local or remote home requests that hit L3 cache in the uncore and was serviced by another core with a cross core snoop where no modified copies were found (clean)",
				.pme_ucode  = 0x200,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "OTHER_CORE_HITM",
				.pme_udesc  = "Response. Counts L3 Hit: local or remote home requests that hit L3 cache in the uncore and was serviced by another core with a cross core snoop where modified copies were found (HITM)",
				.pme_ucode  = 0x400,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "REMOTE_CACHE_FWD",
				.pme_udesc  = "Response. Counts L3 Miss: local homed requests that missed the L3 cache and was serviced by forwarded data following a cross package snoop where no modified copies found. (Remote home requests are not counted)",
				.pme_ucode  = 0x1000,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "REMOTE_DRAM",
				.pme_udesc  = "Response. Counts L3 Miss: remote home requests that missed the L3 cache and were serviced by remote DRAM",
				.pme_ucode  = 0x2000,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "LOCAL_DRAM",
				.pme_udesc  = "Response. Counts L3 Miss: local home requests that missed the L3 cache and were serviced by local DRAM",
				.pme_ucode  = 0x4000,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "NON_DRAM",
				.pme_udesc  = "Response. Non-DRAM requests that were serviced by IOH",
				.pme_ucode  = 0x8000,
				.pme_uflags = 0,
			},
			{ .pme_uname  = "ANY_RESPONSE",
				.pme_udesc  = "Response. Counts any response type",
				.pme_ucode  = 0xf700,
				.pme_uflags = 0,
			},
		},
		.pme_numasks = 17
	}
};
#define PME_COREI7_UNHALTED_CORE_CYCLES 0
#define PME_COREI7_INSTRUCTIONS_RETIRED 1
#define PME_COREI7_EVENT_COUNT	  (sizeof(corei7_pe)/sizeof(pme_nhm_entry_t))
