# -*- coding: utf-8 -*-

#
# Copyright (c) 2012 Red Hat, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#           http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
from optparse import Option
from redhat_support_tool.helpers.confighelper import EmptyValueError, _
from redhat_support_tool.plugins import Plugin
from redhat_support_tool.helpers.yumdownloadhelper import YumDownloadHelper, \
                                                          NoReposError
import logging
import redhat_support_tool.helpers.confighelper as confighelper
import os


__author__ = 'Nigel Jones <nigjones@redhat.com>'
__author__ = 'Keith Robertson <kroberts@redhat.com>'
logger = logging.getLogger("redhat_support_tool.plugins.getkerneldebug")


class GetKernelDebugPackages(Plugin):
    plugin_name = 'getkerneldebug'
    yumhelper = None
    pkgAry = []
    yumquery = None

    @classmethod
    def get_usage(cls):
        '''
        The usage statement that will be printed by OptionParser.

        Example:
            - %prog -c CASENUMBER [options] <comment text here>
        Important: %prog is a OptionParser built-in.  Use it!
        '''
        return _('%prog <package name>')

    @classmethod
    def get_desc(cls):
        '''
        The description statement that will be printed by OptionParser.

        Example:
            - 'Use the \'%s\' command to add a comment to a case.'\
             % cls.plugin_name
        '''
        return _('Use the \'%s\' command to download available debug '
                 'vmlinux images.  Wildcards are allowed.  (requires root user '
                 'privileges)') % cls.plugin_name

    @classmethod
    def get_epilog(cls):
        '''
        The epilog string that will be printed by OptionParser.  Usually
        used to print an example of how to use the program.

        Example:
         Examples:
          - %s -c 12345678 Lorem ipsum dolor sit amet, consectetur adipisicing
          - %s -c 12345678
        '''
        return _("""Examples:
  - %s 2.6.32-343.el6
  - %s 2.6.18-128.*
  - %s -t xen 2.6.18-348.el5""") \
  % (cls.plugin_name, cls.plugin_name, cls.plugin_name)

    @classmethod
    def get_options(cls):
        '''
        Subclasses that need command line options should override this method
        and return an array of optparse.Option(s) to be used by the
        OptionParser.

        Example:
         return [Option("-f", "--file", action="store",
                        dest="filename", help='Some file'),
                 Option("-c", "--case",
                        action="store", dest="casenumber",
                        help='A case')]

         Would produce the following:
         Command (? for help): help mycommand

         Usage: mycommand [options]

         Use the 'mycommand' command to find a knowledge base solution by ID
         Options:
           -h, --help  show this help message and exit
           -f, --file  Some file
           -c, --case  A case
         Example:
          - mycommand -c 12345 -f abc.txt

        '''
        return [Option("-n", "--noprompt", dest="noprompt",
                    action="store_true",
                    help=_('For multi-package installs do not '
                           'ask for confirmation.'),
                    default=False),
                Option("-t", "--variant", dest="variant",
                       help=_('Select an alternative kernel variant'),
                       metavar=_('VARIANT'))]

    def insert_obj(self, yumdict):
        '''
        Allow insertion of a package object by launchhelper (when selecting
        from the list generated by list_kerneldebugs.py)
        '''
        # Expand yumdict into our YumDownloadHelper and package
        self.yumhelper = yumdict['yumhelper']
        # Package is
        self.pkgAry = [yumdict['package']]

    def validate_args(self):
        # Launchhelper may have inserted a package object from earlier
        if not self._args:
            if (len(self.pkgAry) == 0 or self.yumhelper == None):
                msg = _('ERROR: %s requires a package name.') % \
                    GetKernelDebugPackages.plugin_name
                print msg
                raise Exception(msg)
        # Otherwise, build a search string:
        else:
            if self._options['variant']:
                self.yumquery = 'kernel-%s-debuginfo-%s' % \
                    (self._options['variant'], self._line)
            else:
                self.yumquery = 'kernel-debuginfo-%s' % (self._line)

    def postinit(self):
        try:
            if os.geteuid() != 0:
                raise Exception(_('This command requires root user '
                                  'privileges.'))
            if len(self.pkgAry) == 0:
                self.yumhelper = YumDownloadHelper()
                self.yumhelper.setup_debug_repos()
                self.pkgAry = self.yumhelper.find_package(self.yumquery)
                if not self.pkgAry:
                    raise EmptyValueError(
                            _('%s was not available from any of the following'
                              ' yum repositories: %s') % (self.yumquery,
                                    ', '.join(self.yumhelper.get_repoids())))
        except NoReposError, nre:
            print nre
            raise
        except EmptyValueError, eve:
            print eve
            raise
        except Exception, e:
            msg = _("ERROR: Unable to get debug packages.  %s") % e
            print msg
            logger.log(logging.ERROR, msg)
            raise

    def non_interactive_action(self):
        try:
            if len(self.pkgAry) >= 1:
                print _('The vmlinux debug images from the following packages '
                        'will be extracted to: %s') % \
                        confighelper.get_config_helper().get(
                                            option='kern_debug_dir')
                for pkg in self.pkgAry:
                    print pkg
                if not self._options['noprompt']:
                    line = raw_input(_('Continue (y/n)? '))
                    if str(line).strip().lower() != 'y':
                        print _('Canceling')
                        return
                # Start installing'
                extpaths = self.yumhelper.extractKernelDebugs(self.pkgAry)
                for extractedpkg in extpaths:
                    print _('Kernel vmlinux file for %s was '
                            'extracted to %s') % (extractedpkg['package'],
                                                  extractedpkg['path'])
            else:
                print _('No packages to install.')
        # pylint: disable=W0703
        except Exception, e:
            logger.log(logging.WARNING, e)
