// +build go1.10

/* Copyright 2018 The Bazel Authors. All rights reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package buildid_test

import (
	"bytes"
	"errors"
	"os"
	"os/exec"
	"path/filepath"
	"testing"
)

func TestEmptyBuildID(t *testing.T) {
	// Locate the buildid tool and several archive files to check.
	//   fmt.a - pure go
	//   crypto/aes.a - contains assembly
	//   runtime/cgo.a - contains cgo
	// The path may vary depending on platform and architecture, so just
	// do a search.
	var buildidPath string
	pkgPaths := map[string]string{
		"fmt.a": "",
		"aes.a": "",
		"cgo.a": "",
	}
	n := len(pkgPaths)
	done := errors.New("done")
	var visit filepath.WalkFunc
	visit = func(path string, info os.FileInfo, err error) error {
		if err != nil {
			return err
		}
		if (info.Mode() & os.ModeType) == os.ModeSymlink {
			path, err = filepath.EvalSymlinks(path)
			if err != nil {
				return err
			}
			return filepath.Walk(path, visit)
		}
		if filepath.Base(path) == "buildid" && (info.Mode()&0111) != 0 {
			buildidPath = path
		}
		for pkg := range pkgPaths {
			if filepath.Base(path) == pkg {
				pkgPaths[pkg] = path
				n--
			}
		}
		if buildidPath != "" && n == 0 {
			return done
		}
		return nil
	}
	if err := filepath.Walk(".", visit); err == nil {
		t.Fatal("could not locate stdlib ROOT file")
	} else if err != done {
		t.Fatal(err)
	}
	if buildidPath == "" {
		t.Fatal("buildid not found")
	}

	for pkg, path := range pkgPaths {
		if path == "" {
			t.Errorf("could not locate %s", pkg)
			continue
		}
		// Equivalent to: go tool buildid pkg.a
		// It's an error if this produces any output.
		cmd := exec.Command(buildidPath, path)
		out, err := cmd.Output()
		if err != nil {
			t.Error(err)
		}
		if len(bytes.TrimSpace(out)) > 0 {
			t.Errorf("%s: unexpected buildid: %s", path, out)
		}
	}
}
